<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Api\VerifyGuestPinRequest;
use App\Models\Event;
use App\Support\GuestTokenService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\RateLimiter;

class EventShareController extends ApiController
{
    public function show(Request $request, Event $event)
    {
        if (! $this->tokenMatches($request, $event)) {
            abort(404);
        }

        if (! $event->is_active) {
            return $this->success([
                'event' => $event,
                'inactive' => true,
                'expired' => $this->isExpired($event),
                'pin_required' => false,
                'media' => [],
            ]);
        }

        $pinRequired = $this->requiresPin($event);
        $media = [];
        $stats = $this->emptyMediaStats();

        if (! $pinRequired) {
            $media = $event->media()
                ->whereIn('status', ['ready', 'processing', 'pending'])
                ->orderByDesc('created_at')
                ->get();

            $stats = [
                'total' => $media->count(),
                'ready' => $media->where('status', 'ready')->count(),
                'processing' => $media->where('status', 'processing')->count(),
                'pending' => $media->where('status', 'pending')->count(),
            ];
        }

        return $this->success([
            'event' => $event,
            'inactive' => false,
            'expired' => $this->isExpired($event),
            'pin_required' => $pinRequired,
            'media' => $media,
            'media_stats' => $stats,
        ]);
    }

    public function verifyPin(VerifyGuestPinRequest $request, Event $event, GuestTokenService $tokens)
    {
        if (! $this->tokenMatches($request, $event)) {
            abort(404);
        }

        $throttleKey = 'event-pin|' . $event->id . '|' . $request->ip();

        if (RateLimiter::tooManyAttempts($throttleKey, 5)) {
            $seconds = RateLimiter::availableIn($throttleKey);

            return $this->success([
                'message' => 'Too many attempts. Try again later.',
                'retry_after' => $seconds,
            ], [], 429);
        }

        if (! $this->pinMatches($event, $request->input('pin'))) {
            RateLimiter::hit($throttleKey, 60);

            return $this->success([
                'message' => 'Invalid pin.',
            ], [], 422);
        }

        RateLimiter::clear($throttleKey);

        $token = $tokens->issue($event->id, 30);

        return $this->success([
            'guest_token' => $token,
            'expires_in' => 30 * 60,
        ]);
    }

    private function tokenMatches(Request $request, Event $event): bool
    {
        $token = (string) $request->query('token', $request->input('token', ''));

        return $token !== '' && hash_equals($event->share_token ?? '', $token);
    }

    private function requiresPin(Event $event): bool
    {
        return (bool) ($event->guest_pin || $event->admin_pin);
    }

    private function pinMatches(Event $event, string $pin): bool
    {
        foreach (['guest_pin', 'admin_pin'] as $field) {
            $encrypted = $event->{$field};
            if (! $encrypted) {
                continue;
            }

            try {
                $decrypted = Crypt::decryptString($encrypted);
            } catch (\Throwable $exception) {
                continue;
            }

            if (hash_equals($decrypted, $pin)) {
                return true;
            }
        }

        return false;
    }

    private function isExpired(Event $event): bool
    {
        if (! $event->expiry_date) {
            return false;
        }

        return $event->expiry_date->isPast();
    }

    private function emptyMediaStats(): array
    {
        return [
            'total' => 0,
            'ready' => 0,
            'processing' => 0,
            'pending' => 0,
        ];
    }
}
