<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportEventsRequest;
use App\Http\Requests\Admin\StoreEventRequest;
use App\Http\Requests\Admin\UpdateEventRequest;
use App\Models\Event;
use App\Services\ExportService;
use App\Services\PlanLimitService;
use App\Exceptions\PlanLimitReachedException;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Str;

class EventsController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', Event::class);

        $search = $request->query('search');
        $type = $request->query('type');
        $status = $request->query('status');
        $sort = $request->query('sort', 'date_desc');

        $query = Event::query()->withCount('media')->with('coverMedia');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('type', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($type) {
            $query->where('type', $type);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreEventRequest $request, PlanLimitService $planLimits)
    {
        $this->authorize('create', Event::class);

        $data = $request->validated();
        $user = $request->user();

        if ($user) {
            try {
                $planLimits->assertCanCreateEvent($user, $request->boolean('is_active'));
            } catch (PlanLimitReachedException $exception) {
                return $this->success([
                    'message' => $exception->getMessage(),
                ], [], 422);
            }
        }

        $event = Event::create([
            'name' => $data['name'],
            'slug' => $this->uniqueSlug($data['name']),
            'type' => $data['type'],
            'details' => $data['details'] ?? null,
            'event_date' => $data['event_date'],
            'expiry_date' => $data['expiry_date'] ?? null,
            'notifications_enabled' => $request->boolean('notifications_enabled'),
            'is_active' => $request->boolean('is_active'),
            'hashtags' => $this->normalizeTags($data['hashtags'] ?? null),
            'share_token' => $this->uniqueToken(),
            'created_by' => $request->user()?->id,
            'guest_pin' => $request->filled('guest_pin') ? Crypt::encryptString($request->input('guest_pin')) : null,
            'admin_pin' => $request->filled('admin_pin') ? Crypt::encryptString($request->input('admin_pin')) : null,
            'watermark_position' => $user && ($user->hasRole(['Super Admin', 'Admin']) || $user->plan?->has_watermark) ? (($data['watermark_position'] ?? null) ?: null) : null,
        ]);

        return $this->success($event->fresh()->load('coverMedia'), [], 201);
    }

    public function show(Event $event)
    {
        $this->authorize('view', $event);

        $event->load('coverMedia');

        $mediaCounts = [
            'total' => $event->media()->count(),
            'images' => $event->media()->where('file_type', 'image')->count(),
            'videos' => $event->media()->where('file_type', 'video')->count(),
        ];

        $pipeline = [
            'pending' => $event->media()->where('status', 'pending')->count(),
            'processing' => $event->media()->where('status', 'processing')->count(),
            'ready' => $event->media()->where('status', 'ready')->count(),
            'failed' => $event->media()->where('status', 'failed')->count(),
        ];

        return $this->success([
            'event' => $event,
            'media_counts' => $mediaCounts,
            'media_pipeline' => $pipeline,
            'share_url' => $event->share_url,
        ]);
    }

    public function update(UpdateEventRequest $request, Event $event, PlanLimitService $planLimits)
    {
        $this->authorize('update', $event);

        $data = $request->validated();
        $user = $request->user();

        if ($user && ! $event->is_active && $request->boolean('is_active')) {
            try {
                $planLimits->assertCanCreateEvent($user, true);
            } catch (PlanLimitReachedException $exception) {
                return $this->success([
                    'message' => $exception->getMessage(),
                ], [], 422);
            }
        }

        $payload = [
            'name' => $data['name'],
            'type' => $data['type'],
            'details' => $data['details'] ?? null,
            'event_date' => $data['event_date'],
            'expiry_date' => $data['expiry_date'] ?? null,
            'notifications_enabled' => $request->boolean('notifications_enabled'),
            'is_active' => $request->boolean('is_active'),
            'hashtags' => $this->normalizeTags($data['hashtags'] ?? null),
        ];

        if ($user && ($user->hasRole(['Super Admin', 'Admin']) || $user->plan?->has_watermark)) {
            $payload['watermark_position'] = ($data['watermark_position'] ?? null) ?: null;
        }

        if ($data['name'] !== $event->name) {
            $payload['slug'] = $this->uniqueSlug($data['name'], $event->id);
        }

        if ($request->filled('guest_pin')) {
            $payload['guest_pin'] = Crypt::encryptString($request->input('guest_pin'));
        }

        if ($request->filled('admin_pin')) {
            $payload['admin_pin'] = Crypt::encryptString($request->input('admin_pin'));
        }

        $event->update($payload);

        return $this->success($event->fresh()->load('coverMedia'));
    }

    public function stats(Event $event)
    {
        $this->authorize('view', $event);

        $mediaCounts = [
            'total' => $event->media()->count(),
            'images' => $event->media()->where('file_type', 'image')->count(),
            'videos' => $event->media()->where('file_type', 'video')->count(),
        ];

        $pipeline = [
            'pending' => $event->media()->where('status', 'pending')->count(),
            'processing' => $event->media()->where('status', 'processing')->count(),
            'ready' => $event->media()->where('status', 'ready')->count(),
            'failed' => $event->media()->where('status', 'failed')->count(),
        ];

        $totalSize = (int) $event->media()->sum('size');

        return $this->success([
            'media_counts' => $mediaCounts,
            'media_pipeline' => $pipeline,
            'total_size' => $totalSize,
            'total_size_label' => \App\Services\StorageUsageService::formatBytes($totalSize),
        ]);
    }

    public function destroy(Event $event)
    {
        $this->authorize('delete', $event);

        $event->media()->chunkById(100, function ($media) {
            foreach ($media as $m) {
                $m->delete();
            }
        });
        $event->delete();

        return $this->success(['message' => 'Event deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', Event::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:events,id'],
        ]);

        $events = Event::query()->whereIn('id', $validated['ids'])->get();

        foreach ($events as $event) {
            $event->media()->chunkById(100, function ($media) {
                foreach ($media as $m) {
                    $m->delete();
                }
            });
            $event->delete();
        }

        return $this->success(['message' => 'Selected events deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', Event::class);

        $search = $request->query('search');
        $type = $request->query('type');
        $status = $request->query('status');
        $sort = $request->query('sort', 'date_desc');

        $query = Event::query()->withCount('media');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('type', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($type) {
            $query->where('type', $type);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'name', 'label' => 'Name'],
            ['key' => 'slug', 'label' => 'Slug'],
            ['key' => 'type', 'label' => 'Type'],
            ['key' => 'event_date', 'label' => 'Event Date'],
            ['key' => 'expiry_date', 'label' => 'Expiry Date'],
            ['key' => 'notifications_enabled', 'label' => 'Notifications'],
            ['key' => 'is_active', 'label' => 'Active'],
            ['key' => 'hashtags', 'label' => 'Hashtags'],
            ['key' => 'media_count', 'label' => 'Media Count'],
        ];

        $map = fn (Event $event) => [
            'name' => $event->name,
            'slug' => $event->slug,
            'type' => $event->type,
            'event_date' => optional($event->event_date)->format('Y-m-d'),
            'expiry_date' => optional($event->expiry_date)->format('Y-m-d'),
            'notifications_enabled' => $event->notifications_enabled ? 'Yes' : 'No',
            'is_active' => $event->is_active ? 'Yes' : 'No',
            'hashtags' => $event->hashtags ? implode('|', $event->hashtags) : '',
            'media_count' => $event->media_count,
        ];

        $filename = 'events-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Events', 'Event export');
    }

    public function import(ImportEventsRequest $request)
    {
        $this->authorize('import', Event::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, [
            'name',
            'type',
            'event_date',
            'expiry_date',
            'notifications_enabled',
            'is_active',
            'hashtags',
        ]);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));
            $type = $this->normalizeType($data['type'] ?? null);
            $eventDate = $this->parseDate($data['event_date'] ?? null);

            if ($name === '' || ! $type || ! $eventDate) {
                $skipped++;
                continue;
            }

            $expiryDate = $this->parseDate($data['expiry_date'] ?? null);
            $hashtags = $this->normalizeTags($data['hashtags'] ?? null);

            $event = Event::query()
                ->where('name', $name)
                ->whereDate('event_date', $eventDate)
                ->first();

            if ($event) {
                $event->update([
                    'type' => $type,
                    'expiry_date' => $expiryDate,
                    'notifications_enabled' => $this->parseBool($data['notifications_enabled'] ?? null, true),
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                    'hashtags' => $hashtags,
                ]);
            } else {
                Event::create([
                    'name' => $name,
                    'slug' => $this->uniqueSlug($name),
                    'type' => $type,
                    'event_date' => $eventDate,
                    'expiry_date' => $expiryDate,
                    'notifications_enabled' => $this->parseBool($data['notifications_enabled'] ?? null, true),
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                    'hashtags' => $hashtags,
                    'share_token' => $this->uniqueToken(),
                ]);
            }

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function normalizeTags(?string $value): ?array
    {
        if (! $value) {
            return null;
        }

        $tags = collect(preg_split('/[|,]/', $value))
            ->map(fn ($tag) => trim($tag))
            ->filter()
            ->unique()
            ->values()
            ->all();

        return $tags ?: null;
    }

    private function uniqueSlug(string $name, ?int $ignoreId = null): string
    {
        $base = Str::slug($name);
        $slug = $base !== '' ? $base : Str::random(8);
        $counter = 1;

        while ($this->slugExists($slug, $ignoreId)) {
            $slug = $base . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    private function slugExists(string $slug, ?int $ignoreId): bool
    {
        $query = Event::query()->where('slug', $slug);
        if ($ignoreId) {
            $query->where('id', '!=', $ignoreId);
        }

        return $query->exists();
    }

    private function uniqueToken(): string
    {
        do {
            $token = Str::random(32);
        } while (Event::query()->where('share_token', $token)->exists());

        return $token;
    }

    private function normalizeType(?string $value): ?string
    {
        if (! $value) {
            return null;
        }

        $types = config('events.types', []);
        foreach ($types as $type) {
            if (Str::lower($type) === Str::lower(trim($value))) {
                return $type;
            }
        }

        return null;
    }

    private function parseDate(?string $value): ?string
    {
        if (! $value) {
            return null;
        }

        try {
            return Carbon::parse($value)->format('Y-m-d');
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_asc' => $query->orderBy('name'),
            'name_desc' => $query->orderBy('name', 'desc'),
            'date_asc' => $query->orderBy('event_date'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('event_date', 'desc'),
        };
    }
}
