<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\UpdateExtensionsRequest;
use App\Models\SystemSetting;
use Illuminate\Http\Request;

class ExtensionsController extends ApiController
{
    public function show()
    {
        $this->authorize('viewAny', SystemSetting::class);

        $setting = SystemSetting::query()->first();
        $payload = $setting?->payload ?? [];

        $defaults = [
            'analytics' => [
                'google_enabled' => false,
                'google_measurement_id' => '',
            ],
            'adsense' => [
                'adsense_enabled' => false,
                'adsense_client_id' => '',
            ],
            'facebook' => [
                'pixel_enabled' => false,
                'pixel_id' => '',
            ],
            'microsoft' => [
                'uet_enabled' => false,
                'uet_tag_id' => '',
            ],
            'social' => [
                'og_enabled' => false,
                'og_title' => '',
                'og_description' => '',
                'og_image' => null,
            ],
        ];

        $extensions = array_replace_recursive($defaults, $payload['extensions'] ?? []);

        return $this->success([
            'settings' => $extensions,
        ]);
    }

    public function update(UpdateExtensionsRequest $request, string $group)
    {
        $this->authorize('update', SystemSetting::class);

        $setting = SystemSetting::query()->firstOrCreate([]);
        $payload = $setting->payload ?? [];
        $extensions = $payload['extensions'] ?? [];

        $data = $request->validated();

        switch ($group) {
            case 'analytics':
                $data['google_enabled'] = $request->boolean('google_enabled');
                break;
            case 'adsense':
                $data['adsense_enabled'] = $request->boolean('adsense_enabled');
                break;
            case 'facebook':
                $data['pixel_enabled'] = $request->boolean('pixel_enabled');
                break;
            case 'microsoft':
                $data['uet_enabled'] = $request->boolean('uet_enabled');
                break;
            case 'social':
                $data['og_enabled'] = $request->boolean('og_enabled');
                $data = $this->handleUpload($request, $data, 'og_image', 'settings/extensions');
                break;
        }

        $extensions[$group] = array_merge($extensions[$group] ?? [], $data);
        $payload['extensions'] = $extensions;
        $setting->payload = $payload;
        $setting->updated_by = $request->user()?->id;
        $setting->save();

        return $this->success([
            'message' => 'Extension settings updated.',
            'settings' => $setting->payload['extensions'] ?? [],
        ]);
    }

    private function handleUpload(Request $request, array $data, string $field, string $path): array
    {
        if ($request->hasFile($field)) {
            $data[$field] = $request->file($field)->store($path, 'public');
        }

        return $data;
    }
}
