<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportFactStatsRequest;
use App\Http\Requests\Admin\StoreFactStatRequest;
use App\Http\Requests\Admin\UpdateFactStatRequest;
use App\Models\FactStat;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class FactStatController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', FactStat::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = FactStat::query()->with('creator');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('fact', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('value', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreFactStatRequest $request)
    {
        $this->authorize('create', FactStat::class);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');
        $data['created_by'] = $request->user()?->id;

        $fact = FactStat::create($data);

        return $this->success($fact->load('creator'), [], 201);
    }

    public function show(FactStat $factStat)
    {
        $this->authorize('view', $factStat);

        return $this->success($factStat->load('creator'));
    }

    public function update(UpdateFactStatRequest $request, FactStat $factStat)
    {
        $this->authorize('update', $factStat);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $factStat->update($data);

        return $this->success($factStat->fresh()->load('creator'));
    }

    public function destroy(FactStat $factStat)
    {
        $this->authorize('delete', $factStat);

        $factStat->delete();

        return $this->success(['message' => 'Fact deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', FactStat::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:fact_stats,id'],
        ]);

        FactStat::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected facts deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', FactStat::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = FactStat::query()->with('creator');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('fact', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('value', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'fact', 'label' => 'Fact'],
            ['key' => 'value', 'label' => 'Value'],
            ['key' => 'is_active', 'label' => 'Status'],
            ['key' => 'created_by', 'label' => 'Created By'],
            ['key' => 'created_at', 'label' => 'Created At'],
        ];

        $map = fn (FactStat $fact) => [
            'fact' => $fact->fact,
            'value' => $fact->value,
            'is_active' => $fact->is_active ? 'Active' : 'Inactive',
            'created_by' => $fact->creator?->email,
            'created_at' => optional($fact->created_at)->format('Y-m-d H:i:s'),
        ];

        $filename = 'fact-stats-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Facts Report', 'Fact export');
    }

    public function import(ImportFactStatsRequest $request)
    {
        $this->authorize('create', FactStat::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, ['fact', 'value', 'is_active']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $fact = trim((string) ($data['fact'] ?? ''));
            $value = trim((string) ($data['value'] ?? ''));

            if ($fact === '' || $value === '') {
                $skipped++;
                continue;
            }

            FactStat::updateOrCreate(
                ['fact' => $fact],
                [
                    'value' => $value,
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'message' => "Imported {$imported} facts. Skipped {$skipped} rows.",
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'fact_asc' => $query->orderBy('fact'),
            'fact_desc' => $query->orderBy('fact', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }
}
