<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportFaqsRequest;
use App\Http\Requests\Admin\StoreFaqRequest;
use App\Http\Requests\Admin\UpdateFaqRequest;
use App\Models\Faq;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class FaqController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', Faq::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'order_asc');

        $query = Faq::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('question', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('answer', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreFaqRequest $request)
    {
        $this->authorize('create', Faq::class);

        $data = $request->validated();

        $faq = Faq::create([
            'question' => $data['question'],
            'slug' => $this->uniqueSlug($data['question']),
            'answer' => $data['answer'],
            'sort_order' => $data['sort_order'] ?? 0,
            'is_active' => $request->boolean('is_active'),
            'created_by' => $request->user()?->id,
        ]);

        return $this->success($faq, [], 201);
    }

    public function show(Faq $faq)
    {
        $this->authorize('view', $faq);

        return $this->success($faq);
    }

    public function update(UpdateFaqRequest $request, Faq $faq)
    {
        $this->authorize('update', $faq);

        $data = $request->validated();

        $payload = [
            'question' => $data['question'],
            'answer' => $data['answer'],
            'sort_order' => $data['sort_order'] ?? 0,
            'is_active' => $request->boolean('is_active'),
        ];

        if ($data['question'] !== $faq->question) {
            $payload['slug'] = $this->uniqueSlug($data['question'], $faq->id);
        }

        $faq->update($payload);

        return $this->success($faq->fresh());
    }

    public function destroy(Faq $faq)
    {
        $this->authorize('delete', $faq);

        $faq->delete();

        return $this->success(['message' => 'FAQ deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', Faq::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:faqs,id'],
        ]);

        Faq::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected FAQs deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', Faq::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'order_asc');

        $query = Faq::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('question', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('answer', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'question', 'label' => 'Question'],
            ['key' => 'answer', 'label' => 'Answer'],
            ['key' => 'sort_order', 'label' => 'Sort Order'],
            ['key' => 'is_active', 'label' => 'Status'],
        ];

        $map = fn (Faq $faq) => [
            'question' => $faq->question,
            'answer' => $faq->answer,
            'sort_order' => $faq->sort_order,
            'is_active' => $faq->is_active ? 'Active' : 'Inactive',
        ];

        $filename = 'faqs-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'FAQ Report', 'FAQ export');
    }

    public function import(ImportFaqsRequest $request)
    {
        $this->authorize('create', Faq::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, ['question', 'answer', 'sort_order', 'is_active']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $question = trim((string) ($data['question'] ?? ''));
            $answer = trim((string) ($data['answer'] ?? ''));

            if ($question === '' || $answer === '') {
                $skipped++;
                continue;
            }

            $faq = Faq::query()->where('question', $question)->first();
            $payload = [
                'answer' => $answer,
                'sort_order' => $this->parseInt($data['sort_order'] ?? null) ?? 0,
                'is_active' => $this->parseBool($data['is_active'] ?? null, true),
            ];

            if ($faq) {
                $faq->update($payload);
            } else {
                Faq::create(array_merge($payload, [
                    'question' => $question,
                    'slug' => $this->uniqueSlug($question),
                ]));
            }

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'message' => "Imported {$imported} FAQs. Skipped {$skipped} rows.",
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function parseInt($value): ?int
    {
        if ($value === null || trim((string) $value) === '') {
            return null;
        }

        return (int) $value;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'order_desc' => $query->orderBy('sort_order', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('sort_order'),
        };
    }

    private function uniqueSlug(string $question, ?int $ignoreId = null): string
    {
        $base = Str::slug($question);
        $slug = $base !== '' ? $base : Str::random(8);
        $counter = 1;

        while ($this->slugExists($slug, $ignoreId)) {
            $slug = $base . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    private function slugExists(string $slug, ?int $ignoreId): bool
    {
        $query = Faq::query()->where('slug', $slug);
        if ($ignoreId) {
            $query->where('id', '!=', $ignoreId);
        }

        return $query->exists();
    }
}
