<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportLanguagesRequest;
use App\Http\Requests\Admin\StoreLanguageRequest;
use App\Http\Requests\Admin\UpdateLanguageRequest;
use App\Models\Language;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class LanguageController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', Language::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = Language::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('code', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('native_name', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreLanguageRequest $request)
    {
        $this->authorize('create', Language::class);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $language = Language::create($data);

        return $this->success($language, [], 201);
    }

    public function show(Language $language)
    {
        $this->authorize('view', $language);

        return $this->success($language);
    }

    public function update(UpdateLanguageRequest $request, Language $language)
    {
        $this->authorize('update', $language);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $language->update($data);

        return $this->success($language->fresh());
    }

    public function destroy(Language $language)
    {
        $this->authorize('delete', $language);

        $language->delete();

        return $this->success(['message' => 'Language deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', Language::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:languages,id'],
        ]);

        Language::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected languages deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', Language::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = Language::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('code', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('native_name', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'name', 'label' => 'Name'],
            ['key' => 'code', 'label' => 'Code'],
            ['key' => 'native_name', 'label' => 'Native Name'],
            ['key' => 'direction', 'label' => 'Direction'],
            ['key' => 'is_active', 'label' => 'Active'],
        ];

        $map = fn (Language $language) => [
            'name' => $language->name,
            'code' => $language->code,
            'native_name' => $language->native_name,
            'direction' => $language->direction,
            'is_active' => $language->is_active ? 'Yes' : 'No',
        ];

        $filename = 'languages-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Languages', 'Language export');
    }

    public function import(ImportLanguagesRequest $request)
    {
        $this->authorize('import', Language::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, ['name', 'code', 'native_name', 'direction', 'is_active']);
        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));
            $code = trim((string) ($data['code'] ?? ''));

            if ($name === '' || $code === '') {
                $skipped++;
                continue;
            }

            Language::updateOrCreate(
                ['code' => $code],
                [
                    'name' => $name,
                    'native_name' => trim((string) ($data['native_name'] ?? '')) ?: null,
                    'direction' => trim((string) ($data['direction'] ?? 'ltr')) ?: 'ltr',
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_desc' => $query->orderBy('name', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('name'),
        };
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }
}
