<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportPagesRequest;
use App\Http\Requests\Admin\StorePageRequest;
use App\Http\Requests\Admin\UpdatePageRequest;
use App\Models\Page;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PagesController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', Page::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $type = $request->query('type');
        $sort = $request->query('sort', 'title_asc');

        $query = Page::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('title', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('description', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        if ($type) {
            $query->where('type', $type);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StorePageRequest $request)
    {
        $this->authorize('create', Page::class);

        $data = $request->validated();

        $page = Page::create([
            'title' => $data['title'],
            'slug' => $this->uniqueSlug($data['title']),
            'type' => $data['type'],
            'description' => $data['description'] ?? null,
            'is_active' => $request->boolean('is_active'),
            'created_by' => $request->user()?->id,
        ]);

        return $this->success($page, [], 201);
    }

    public function show(Page $page)
    {
        $this->authorize('view', $page);

        return $this->success($page);
    }

    public function update(UpdatePageRequest $request, Page $page)
    {
        $this->authorize('update', $page);

        $data = $request->validated();

        $payload = [
            'title' => $data['title'],
            'type' => $data['type'],
            'description' => $data['description'] ?? null,
            'is_active' => $request->boolean('is_active'),
            'updated_by' => $request->user()?->id,
        ];

        if ($data['title'] !== $page->title) {
            $payload['slug'] = $this->uniqueSlug($data['title'], $page->id);
        }

        $page->update($payload);

        return $this->success($page->fresh());
    }

    public function destroy(Page $page)
    {
        $this->authorize('delete', $page);

        $page->delete();

        return $this->success(['message' => 'Page deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', Page::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:pages,id'],
        ]);

        Page::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected pages deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', Page::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $type = $request->query('type');
        $sort = $request->query('sort', 'title_asc');

        $query = Page::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('title', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('description', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        if ($type) {
            $query->where('type', $type);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'title', 'label' => 'Title'],
            ['key' => 'slug', 'label' => 'Slug'],
            ['key' => 'type', 'label' => 'Type'],
            ['key' => 'description', 'label' => 'Description'],
            ['key' => 'is_active', 'label' => 'Status'],
        ];

        $map = fn (Page $page) => [
            'title' => $page->title,
            'slug' => $page->slug,
            'type' => $page->type,
            'description' => $page->description,
            'is_active' => $page->is_active ? 'Active' : 'Inactive',
        ];

        $filename = 'pages-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Pages Report', 'Pages export');
    }

    public function import(ImportPagesRequest $request)
    {
        $this->authorize('create', Page::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, ['title', 'type', 'description', 'is_active']);
        $types = array_keys(config('system.page_types', []));

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $title = trim((string) ($data['title'] ?? ''));

            if ($title === '') {
                $skipped++;
                continue;
            }

            $type = $this->normalizeType($data['type'] ?? null, $types);

            $page = Page::query()->where('title', $title)->first();
            $payload = [
                'type' => $type,
                'description' => $data['description'] ?? null,
                'is_active' => $this->parseBool($data['is_active'] ?? null, true),
            ];

            if ($page) {
                $page->update($payload);
            } else {
                Page::create(array_merge($payload, [
                    'title' => $title,
                    'slug' => $this->uniqueSlug($title),
                ]));
            }

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'message' => "Imported {$imported} pages. Skipped {$skipped} rows.",
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function normalizeType(?string $value, array $types): string
    {
        $raw = trim((string) $value);
        $normalized = Str::slug($raw, '_');

        if ($normalized === 'terms_and_conditions') {
            $normalized = 'terms';
        }

        if (in_array($normalized, $types, true)) {
            return $normalized;
        }

        foreach (config('system.page_types', []) as $key => $label) {
            if (Str::lower($label) === Str::lower($raw)) {
                return $key;
            }
        }

        return 'other';
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'title_desc' => $query->orderBy('title', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('title'),
        };
    }

    private function uniqueSlug(string $title, ?int $ignoreId = null): string
    {
        $base = Str::slug($title);
        $slug = $base !== '' ? $base : Str::random(8);
        $counter = 1;

        while ($this->slugExists($slug, $ignoreId)) {
            $slug = $base . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    private function slugExists(string $slug, ?int $ignoreId): bool
    {
        $query = Page::query()->where('slug', $slug);
        if ($ignoreId) {
            $query->where('id', '!=', $ignoreId);
        }

        return $query->exists();
    }
}
