<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\StorePermissionRequest;
use App\Http\Requests\Admin\UpdatePermissionRequest;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Permission;

class PermissionsController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', Permission::class);

        $search = $request->query('search');

        $query = Permission::query()->orderBy('name');

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StorePermissionRequest $request)
    {
        $this->authorize('create', Permission::class);

        $data = $request->validated();
        $permission = Permission::create($data);

        return $this->success($permission, [], 201);
    }

    public function show(Permission $permission)
    {
        $this->authorize('view', $permission);

        $permission->loadCount('roles');

        return $this->success($permission);
    }

    public function update(UpdatePermissionRequest $request, Permission $permission)
    {
        $this->authorize('update', $permission);

        $data = $request->validated();
        $permission->update($data);

        return $this->success($permission);
    }

    public function destroy(Permission $permission)
    {
        $this->authorize('delete', $permission);

        $permission->delete();

        return $this->success(['message' => 'Permission deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', Permission::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:permissions,id'],
        ]);

        Permission::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected permissions deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', Permission::class);

        $search = $request->query('search');
        $query = Permission::query()->orderBy('name');

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        $columns = [
            ['key' => 'name', 'label' => 'Name'],
            ['key' => 'guard_name', 'label' => 'Guard'],
        ];

        $map = fn (Permission $permission) => [
            'name' => $permission->name,
            'guard_name' => $permission->guard_name,
        ];

        $filename = 'permissions-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Permission Report', 'Permission export');
    }
}
