<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportPlansRequest;
use App\Http\Requests\Admin\StorePlanRequest;
use App\Http\Requests\Admin\UpdatePlanRequest;
use App\Models\Plan;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PlanController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', Plan::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = Plan::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StorePlanRequest $request)
    {
        $this->authorize('create', Plan::class);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');
        $data['slug'] = Str::slug($data['name']);
        $data['created_by'] = $request->user()?->id;

        $plan = Plan::create($data);

        return $this->success($plan, [], 201);
    }

    public function show(Plan $plan)
    {
        $this->authorize('view', $plan);

        return $this->success($plan);
    }

    public function update(UpdatePlanRequest $request, Plan $plan)
    {
        $this->authorize('update', $plan);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        if ($plan->name !== $data['name']) {
            $data['slug'] = Str::slug($data['name']);
        }

        $plan->update($data);

        return $this->success($plan->fresh());
    }

    public function destroy(Plan $plan)
    {
        $this->authorize('delete', $plan);

        $plan->delete();

        return $this->success(['message' => 'Plan deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', Plan::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:plans,id'],
        ]);

        Plan::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected plans deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', Plan::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = Plan::query();

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'name', 'label' => 'Name'],
            ['key' => 'price', 'label' => 'Price'],
            ['key' => 'offer_price', 'label' => 'Offer Price'],
            ['key' => 'duration', 'label' => 'Duration'],
            ['key' => 'images_limit', 'label' => 'Image Limit'],
            ['key' => 'events_limit', 'label' => 'Event Limit'],
            ['key' => 'storage_limit_gb', 'label' => 'Storage Limit (GB)'],
            ['key' => 'status', 'label' => 'Status'],
        ];

        $map = fn (Plan $plan) => [
            'name' => $plan->name,
            'price' => $plan->price == 0 ? 'Free' : (string) $plan->price,
            'offer_price' => $plan->offer_price !== null ? (string) $plan->offer_price : '-',
            'duration' => $plan->duration_type === 'lifetime'
                ? 'Lifetime'
                : ($plan->duration_months ? $plan->duration_months . ' months' : 'Monthly'),
            'images_limit' => $plan->images_limit ?? 'Unlimited',
            'events_limit' => $plan->events_limit ?? 'Unlimited',
            'storage_limit_gb' => $plan->storage_limit_gb ?? 'Unlimited',
            'status' => $plan->is_active ? 'Active' : 'Inactive',
        ];

        $filename = 'plans-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Plans', 'Plan export');
    }

    public function import(ImportPlansRequest $request)
    {
        $this->authorize('import', Plan::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, ['name', 'price', 'offer_price', 'features', 'images_limit', 'events_limit', 'storage_limit_gb', 'duration_type', 'duration_months', 'is_active']);
        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));

            if ($name === '') {
                $skipped++;
                continue;
            }

            Plan::updateOrCreate(
                ['slug' => Str::slug($name)],
                [
                    'name' => $name,
                    'price' => (float) ($data['price'] ?? 0),
                    'offer_price' => $data['offer_price'] !== null ? (float) $data['offer_price'] : null,
                    'features' => $this->parseFeatures($data['features'] ?? null),
                    'images_limit' => $data['images_limit'] !== null ? (int) $data['images_limit'] : null,
                    'events_limit' => $data['events_limit'] !== null ? (int) $data['events_limit'] : null,
                    'storage_limit_gb' => $data['storage_limit_gb'] !== null ? (float) $data['storage_limit_gb'] : null,
                    'duration_type' => $data['duration_type'] ?: 'months',
                    'duration_months' => $data['duration_months'] !== null ? (int) $data['duration_months'] : null,
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_asc' => $query->orderBy('name'),
            'name_desc' => $query->orderBy('name', 'desc'),
            'price_asc' => $query->orderBy('price'),
            'price_desc' => $query->orderBy('price', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function parseFeatures(?string $value): ?array
    {
        if (! $value) {
            return null;
        }

        $items = collect(preg_split('/[|,]/', $value))
            ->map(fn ($item) => trim($item))
            ->filter()
            ->unique()
            ->values()
            ->all();

        return $items ?: null;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }
}
