<?php

namespace App\Http\Controllers\Api;

use App\Models\BlogCategory;
use App\Models\BlogPost;
use App\Models\FactStat;
use App\Models\Faq;
use App\Models\Page;
use App\Models\Plan;
use App\Models\Testimonial;
use Illuminate\Http\Request;

class PublicContentController extends ApiController
{
    public function faqs()
    {
        $items = Faq::query()
            ->where('is_active', true)
            ->orderBy('sort_order')
            ->get();

        return $this->success($items);
    }

    public function testimonials()
    {
        $items = Testimonial::query()
            ->where('is_active', true)
            ->orderByDesc('created_at')
            ->get();

        return $this->success($items);
    }

    public function facts()
    {
        $items = FactStat::query()
            ->where('is_active', true)
            ->orderByDesc('created_at')
            ->get();

        return $this->success($items);
    }

    public function plans()
    {
        $items = Plan::query()
            ->where('is_active', true)
            ->orderBy('price')
            ->get();

        return $this->success($items);
    }

    public function pages(Request $request)
    {
        $type = $request->query('type');

        $query = Page::query()->where('is_active', true);

        if ($type) {
            $query->where('type', $type);
        }

        return $this->success($query->orderBy('title')->get());
    }

    public function page(Request $request, string $slug)
    {
        $type = $request->query('type');

        $page = Page::query()
            ->where('is_active', true)
            ->where('slug', $slug)
            ->first();

        if (! $page && $type) {
            $page = Page::query()
                ->where('is_active', true)
                ->where('type', $type)
                ->first();
        }

        if (! $page) {
            return $this->success(null, [], 404);
        }

        return $this->success($page);
    }

    public function blogCategories()
    {
        $items = BlogCategory::query()
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return $this->success($items);
    }

    public function blogPosts(Request $request)
    {
        $search = $request->query('search');
        $category = $request->query('category');
        $tag = $request->query('tag');
        $sort = $request->query('sort', 'published_desc');
        $perPage = $this->perPage($request->query('per_page'));

        $query = BlogPost::query()
            ->with('category')
            ->where('is_active', true);

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('title', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('excerpt', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('details', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($category) {
            $query->whereHas('category', function ($builder) use ($category) {
                $builder->where('slug', $category)
                    ->orWhere('id', $category);
            });
        }

        if ($tag) {
            $query->whereJsonContains('tags', $tag);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate($perPage)->withQueryString());
    }

    public function blogPost(string $slug)
    {
        $post = BlogPost::query()
            ->with('category')
            ->where('is_active', true)
            ->where('slug', $slug)
            ->first();

        if (! $post) {
            return $this->success(null, [], 404);
        }

        return $this->success($post);
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'published_asc' => $query->orderBy('published_at'),
            'title_asc' => $query->orderBy('title'),
            'title_desc' => $query->orderBy('title', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('published_at', 'desc'),
        };
    }

    private function perPage($value): int
    {
        $perPage = (int) ($value ?: 12);

        if ($perPage < 1) {
            return 12;
        }

        return min($perPage, 50);
    }
}
