<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Api\PublicSupportTicketRequest;
use App\Models\SupportTicket;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;

class PublicSupportController extends ApiController
{
    public function store(PublicSupportTicketRequest $request)
    {
        $data = $request->validated();

        $ticket = SupportTicket::create([
            'ticket_number' => $this->uniqueTicketNumber(),
            'subject' => $data['subject'],
            'message' => $data['message'],
            'requester_name' => $data['requester_name'],
            'requester_email' => $data['requester_email'],
            'requester_phone' => $data['requester_phone'] ?? null,
            'priority' => $this->defaultPriority(),
            'status' => $this->defaultStatus(),
            'is_active' => true,
            'resolved_at' => $this->resolveTimestamp($this->defaultStatus()),
        ]);

        return $this->success([
            'message' => 'Support request submitted.',
            'ticket' => $ticket,
        ], [], 201);
    }

    private function uniqueTicketNumber(): string
    {
        do {
            $number = 'TKT-' . now()->format('Ymd') . '-' . strtoupper(Str::random(6));
        } while (SupportTicket::query()->where('ticket_number', $number)->exists());

        return $number;
    }

    private function resolveTimestamp(string $status, ?Carbon $current = null): ?Carbon
    {
        if (in_array($status, ['resolved', 'closed'], true)) {
            return $current ?? now();
        }

        return null;
    }

    private function defaultPriority(): string
    {
        $priorities = array_keys(config('support.priorities', []));

        return in_array('medium', $priorities, true) ? 'medium' : ($priorities[0] ?? 'medium');
    }

    private function defaultStatus(): string
    {
        $statuses = array_keys(config('support.statuses', []));

        return in_array('open', $statuses, true) ? 'open' : ($statuses[0] ?? 'open');
    }
}
