<?php

namespace App\Http\Controllers\Api;

use App\Models\Coupon;
use App\Models\Event;
use App\Models\Plan;
use App\Models\Report;
use App\Models\SupportTicket;
use App\Models\User;
use App\Services\ExportService;
use App\Support\FormatSettings;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Spatie\Permission\Models\Role;

class ReportController extends ApiController
{
    public function index()
    {
        $this->authorize('viewAny', Report::class);

        $reports = [
            [
                'key' => 'users',
                'title' => 'User Report',
                'description' => 'Track account growth, verification, and assigned roles.',
                'total' => User::query()->count(),
            ],
            [
                'key' => 'events',
                'title' => 'Event Report',
                'description' => 'Review upcoming events, activity, and media volume.',
                'total' => Event::query()->count(),
            ],
            [
                'key' => 'plans',
                'title' => 'Plan Report',
                'description' => 'Monitor plan pricing, limits, and activation status.',
                'total' => Plan::query()->count(),
            ],
            [
                'key' => 'coupons',
                'title' => 'Coupon Report',
                'description' => 'Audit promo performance and validity windows.',
                'total' => Coupon::query()->count(),
            ],
            [
                'key' => 'support-tickets',
                'title' => 'Support Ticket Report',
                'description' => 'Track request volumes, priorities, and resolution status.',
                'total' => SupportTicket::query()->count(),
            ],
        ];

        return $this->success($reports);
    }

    public function show(Request $request, string $report)
    {
        $this->authorize('view', Report::class);

        $definition = $this->definition($report);
        $query = ($definition['query'])();
        $filters = $this->applyFilters($report, $query, $request);

        $rows = $query->paginate(15)->withQueryString()->through($definition['map']);

        return $this->success([
            'report' => [
                'key' => $report,
                'title' => $definition['title'],
                'subtitle' => $definition['subtitle'],
                'description' => $definition['description'],
                'empty' => $definition['empty'],
                'columns' => $definition['columns'],
            ],
            'filters' => $filters,
            'rows' => $rows->items(),
        ], [
            'current_page' => $rows->currentPage(),
            'per_page' => $rows->perPage(),
            'total' => $rows->total(),
            'last_page' => $rows->lastPage(),
        ]);
    }

    public function export(Request $request, string $report, ExportService $exporter)
    {
        $this->authorize('export', Report::class);

        $definition = $this->definition($report);
        $query = ($definition['query'])();
        $this->applyFilters($report, $query, $request);

        $filename = $report . '-' . now()->format('Ymd-His');

        return $exporter->download(
            (string) $request->query('format', 'csv'),
            $filename,
            $query,
            $definition['columns'],
            $definition['map'],
            $definition['title'],
            $definition['description']
        );
    }

    private function definition(string $report): array
    {
        $definitions = $this->definitions();

        if (! array_key_exists($report, $definitions)) {
            abort(404);
        }

        return $definitions[$report];
    }

    private function definitions(): array
    {
        return [
            'users' => [
                'title' => 'User Report',
                'subtitle' => 'Monitor account status, verification, and role coverage.',
                'description' => 'Analyze user growth with status and role distribution.',
                'empty' => 'No users match the selected filters.',
                'columns' => [
                    ['key' => 'name', 'label' => 'Name'],
                    ['key' => 'email', 'label' => 'Email'],
                    ['key' => 'phone', 'label' => 'Phone'],
                    ['key' => 'roles', 'label' => 'Roles'],
                    ['key' => 'status', 'label' => 'Status'],
                    ['key' => 'verified', 'label' => 'Verified'],
                    ['key' => 'created_at', 'label' => 'Joined'],
                ],
                'query' => fn () => User::query()->with('roles')->orderBy('created_at', 'desc'),
                'map' => fn (User $user) => [
                    'name' => $user->name,
                    'email' => $user->email,
                    'phone' => $user->phone ?? '-',
                    'roles' => $user->roles->pluck('name')->implode(', ') ?: 'None',
                    'status' => $user->is_active ? 'Active' : 'Inactive',
                    'verified' => $user->email_verified_at ? 'Yes' : 'No',
                    'created_at' => FormatSettings::date($user->created_at),
                ],
            ],
            'events' => [
                'title' => 'Event Report',
                'subtitle' => 'Track event timelines, privacy status, and uploads.',
                'description' => 'Review event types, dates, and media volume.',
                'empty' => 'No events match the selected filters.',
                'columns' => [
                    ['key' => 'name', 'label' => 'Event'],
                    ['key' => 'type', 'label' => 'Type'],
                    ['key' => 'event_date', 'label' => 'Event Date'],
                    ['key' => 'expiry_date', 'label' => 'Expiry Date'],
                    ['key' => 'media_count', 'label' => 'Media'],
                    ['key' => 'status', 'label' => 'Status'],
                    ['key' => 'created_by', 'label' => 'Created By'],
                ],
                'query' => fn () => Event::query()
                    ->with('createdBy')
                    ->withCount('media')
                    ->orderBy('event_date', 'desc'),
                'map' => fn (Event $event) => [
                    'name' => $event->name,
                    'type' => $event->type,
                    'event_date' => FormatSettings::date($event->event_date),
                    'expiry_date' => FormatSettings::date($event->expiry_date),
                    'media_count' => (string) $event->media_count,
                    'status' => $event->is_active ? 'Active' : 'Inactive',
                    'created_by' => $event->createdBy?->name ?? 'System',
                ],
            ],
            'plans' => [
                'title' => 'Plan Report',
                'subtitle' => 'Review pricing, durations, and included limits.',
                'description' => 'Track active plan offerings and pricing structure.',
                'empty' => 'No plans match the selected filters.',
                'columns' => [
                    ['key' => 'name', 'label' => 'Plan'],
                    ['key' => 'price', 'label' => 'Price'],
                    ['key' => 'offer_price', 'label' => 'Offer'],
                    ['key' => 'duration', 'label' => 'Duration'],
                    ['key' => 'images_limit', 'label' => 'Image Limit'],
                    ['key' => 'events_limit', 'label' => 'Event Limit'],
                    ['key' => 'status', 'label' => 'Status'],
                ],
                'query' => fn () => Plan::query()->orderBy('created_at', 'desc'),
                'map' => fn (Plan $plan) => [
                    'name' => $plan->name,
                    'price' => $plan->price == 0 ? 'Free' : FormatSettings::currency($plan->price),
                    'offer_price' => $plan->offer_price !== null ? FormatSettings::currency($plan->offer_price) : '-',
                    'duration' => $plan->duration_type === 'lifetime'
                        ? 'Lifetime'
                        : ($plan->duration_months ? $plan->duration_months . ' months' : 'Monthly'),
                    'images_limit' => $plan->images_limit !== null ? (string) $plan->images_limit : 'Unlimited',
                    'events_limit' => $plan->events_limit !== null ? (string) $plan->events_limit : 'Unlimited',
                    'status' => $plan->is_active ? 'Active' : 'Inactive',
                ],
            ],
            'coupons' => [
                'title' => 'Coupon Report',
                'subtitle' => 'Audit promo coverage, timing, and eligibility.',
                'description' => 'Track discount codes and validation windows.',
                'empty' => 'No coupons match the selected filters.',
                'columns' => [
                    ['key' => 'name', 'label' => 'Coupon'],
                    ['key' => 'code', 'label' => 'Code'],
                    ['key' => 'discount_type', 'label' => 'Type'],
                    ['key' => 'discount_value', 'label' => 'Discount'],
                    ['key' => 'min_amount', 'label' => 'Min Amount'],
                    ['key' => 'starts_at', 'label' => 'Starts'],
                    ['key' => 'ends_at', 'label' => 'Ends'],
                    ['key' => 'new_users_only', 'label' => 'New Users'],
                    ['key' => 'status', 'label' => 'Status'],
                ],
                'query' => fn () => Coupon::query()->orderBy('created_at', 'desc'),
                'map' => fn (Coupon $coupon) => [
                    'name' => $coupon->name,
                    'code' => $coupon->code,
                    'discount_type' => $coupon->discount_type === 'percent' ? 'Percent' : 'Fixed',
                    'discount_value' => $coupon->discount_type === 'percent'
                        ? rtrim(rtrim(number_format($coupon->discount_value, 2), '0'), '.') . '%'
                        : FormatSettings::currency($coupon->discount_value),
                    'min_amount' => FormatSettings::currency($coupon->min_amount),
                    'starts_at' => FormatSettings::date($coupon->starts_at),
                    'ends_at' => FormatSettings::date($coupon->ends_at),
                    'new_users_only' => $coupon->new_users_only ? 'Yes' : 'No',
                    'status' => $coupon->is_active ? 'Active' : 'Inactive',
                ],
            ],
            'support-tickets' => [
                'title' => 'Support Ticket Report',
                'subtitle' => 'Monitor incoming requests and response timelines.',
                'description' => 'Review status and assignment coverage.',
                'empty' => 'No tickets match the selected filters.',
                'columns' => [
                    ['key' => 'ticket_number', 'label' => 'Ticket'],
                    ['key' => 'subject', 'label' => 'Subject'],
                    ['key' => 'priority', 'label' => 'Priority'],
                    ['key' => 'status', 'label' => 'Status'],
                    ['key' => 'requester', 'label' => 'Requester'],
                    ['key' => 'requester_email', 'label' => 'Email'],
                    ['key' => 'assigned_to', 'label' => 'Assigned To'],
                    ['key' => 'created_at', 'label' => 'Created'],
                    ['key' => 'resolved_at', 'label' => 'Resolved'],
                ],
                'query' => fn () => SupportTicket::query()
                    ->with('currentAssignment.assignedTo')
                    ->orderBy('created_at', 'desc'),
                'map' => fn (SupportTicket $ticket) => [
                    'ticket_number' => $ticket->ticket_number,
                    'subject' => $ticket->subject,
                    'priority' => $this->formatTitle($ticket->priority),
                    'status' => $this->formatTitle($ticket->status),
                    'requester' => $ticket->requester_name,
                    'requester_email' => $ticket->requester_email,
                    'assigned_to' => $ticket->currentAssignment?->assignedTo?->name ?? 'Unassigned',
                    'created_at' => FormatSettings::date($ticket->created_at),
                    'resolved_at' => FormatSettings::dateTime($ticket->resolved_at),
                ],
            ],
        ];
    }

    private function applyFilters(string $report, Builder $query, Request $request): array
    {
        $search = trim((string) $request->query('search', ''));
        $status = $request->query('status');
        $startDate = $this->normalizeDate($request->query('start_date'));
        $endDate = $this->normalizeDate($request->query('end_date'));

        $fields = [];

        switch ($report) {
            case 'users':
                $role = $request->query('role');

                if ($search !== '') {
                    $query->where(function ($builder) use ($search) {
                        $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('email', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('phone', 'like', "%" . $this->escapeLike($search) . "%");
                    });
                }

                if ($status === 'active') {
                    $query->where('is_active', true);
                } elseif ($status === 'inactive') {
                    $query->where('is_active', false);
                }

                if ($role) {
                    $query->whereHas('roles', function ($builder) use ($role) {
                        $builder->where('id', $role);
                    });
                }

                $this->applyDateRange($query, 'created_at', $startDate, $endDate);

                $roles = Role::query()->orderBy('name')->get()->pluck('name', 'id')->all();
                $roleOptions = array_merge(['' => 'All roles'], $roles);

                $fields = [
                    [
                        'name' => 'search',
                        'type' => 'text',
                        'label' => 'Search',
                        'placeholder' => 'Search by name, email, or phone',
                        'value' => $search,
                    ],
                    [
                        'name' => 'role',
                        'type' => 'select',
                        'label' => 'Role',
                        'options' => $roleOptions,
                        'value' => $role,
                    ],
                    [
                        'name' => 'status',
                        'type' => 'select',
                        'label' => 'Status',
                        'options' => [
                            '' => 'All statuses',
                            'active' => 'Active',
                            'inactive' => 'Inactive',
                        ],
                        'value' => $status,
                    ],
                    [
                        'name' => 'start_date',
                        'type' => 'date',
                        'label' => 'From',
                        'value' => $startDate,
                    ],
                    [
                        'name' => 'end_date',
                        'type' => 'date',
                        'label' => 'To',
                        'value' => $endDate,
                    ],
                ];
                break;
            case 'events':
                $type = $request->query('type');

                if ($search !== '') {
                    $query->where(function ($builder) use ($search) {
                        $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%");
                    });
                }

                if ($type) {
                    $query->where('type', $type);
                }

                if ($status === 'active') {
                    $query->where('is_active', true);
                } elseif ($status === 'inactive') {
                    $query->where('is_active', false);
                }

                $this->applyDateRange($query, 'event_date', $startDate, $endDate);

                $typeOptions = ['' => 'All types'];
                foreach (config('events.types', []) as $eventType) {
                    $typeOptions[$eventType] = $eventType;
                }

                $fields = [
                    [
                        'name' => 'search',
                        'type' => 'text',
                        'label' => 'Search',
                        'placeholder' => 'Search by event name or slug',
                        'value' => $search,
                    ],
                    [
                        'name' => 'type',
                        'type' => 'select',
                        'label' => 'Type',
                        'options' => $typeOptions,
                        'value' => $type,
                    ],
                    [
                        'name' => 'status',
                        'type' => 'select',
                        'label' => 'Status',
                        'options' => [
                            '' => 'All statuses',
                            'active' => 'Active',
                            'inactive' => 'Inactive',
                        ],
                        'value' => $status,
                    ],
                    [
                        'name' => 'start_date',
                        'type' => 'date',
                        'label' => 'From',
                        'value' => $startDate,
                    ],
                    [
                        'name' => 'end_date',
                        'type' => 'date',
                        'label' => 'To',
                        'value' => $endDate,
                    ],
                ];
                break;
            case 'plans':
                if ($search !== '') {
                    $query->where(function ($builder) use ($search) {
                        $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('slug', 'like', "%" . $this->escapeLike($search) . "%");
                    });
                }

                if ($status === 'active') {
                    $query->where('is_active', true);
                } elseif ($status === 'inactive') {
                    $query->where('is_active', false);
                }

                $this->applyDateRange($query, 'created_at', $startDate, $endDate);

                $fields = [
                    [
                        'name' => 'search',
                        'type' => 'text',
                        'label' => 'Search',
                        'placeholder' => 'Search by plan name or slug',
                        'value' => $search,
                    ],
                    [
                        'name' => 'status',
                        'type' => 'select',
                        'label' => 'Status',
                        'options' => [
                            '' => 'All statuses',
                            'active' => 'Active',
                            'inactive' => 'Inactive',
                        ],
                        'value' => $status,
                    ],
                    [
                        'name' => 'start_date',
                        'type' => 'date',
                        'label' => 'From',
                        'value' => $startDate,
                    ],
                    [
                        'name' => 'end_date',
                        'type' => 'date',
                        'label' => 'To',
                        'value' => $endDate,
                    ],
                ];
                break;
            case 'coupons':
                $discountType = $request->query('discount_type');

                if ($search !== '') {
                    $query->where(function ($builder) use ($search) {
                        $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('code', 'like', "%" . $this->escapeLike($search) . "%");
                    });
                }

                if ($discountType) {
                    $query->where('discount_type', $discountType);
                }

                if ($status === 'active') {
                    $query->where('is_active', true);
                } elseif ($status === 'inactive') {
                    $query->where('is_active', false);
                }

                if ($startDate) {
                    $query->whereDate('starts_at', '>=', $startDate);
                }

                if ($endDate) {
                    $query->whereDate('ends_at', '<=', $endDate);
                }

                $fields = [
                    [
                        'name' => 'search',
                        'type' => 'text',
                        'label' => 'Search',
                        'placeholder' => 'Search by coupon name or code',
                        'value' => $search,
                    ],
                    [
                        'name' => 'discount_type',
                        'type' => 'select',
                        'label' => 'Type',
                        'options' => [
                            '' => 'All types',
                            'percent' => 'Percent',
                            'fixed' => 'Fixed',
                        ],
                        'value' => $discountType,
                    ],
                    [
                        'name' => 'status',
                        'type' => 'select',
                        'label' => 'Status',
                        'options' => [
                            '' => 'All statuses',
                            'active' => 'Active',
                            'inactive' => 'Inactive',
                        ],
                        'value' => $status,
                    ],
                    [
                        'name' => 'start_date',
                        'type' => 'date',
                        'label' => 'Valid From',
                        'value' => $startDate,
                    ],
                    [
                        'name' => 'end_date',
                        'type' => 'date',
                        'label' => 'Valid To',
                        'value' => $endDate,
                    ],
                ];
                break;
            case 'support-tickets':
                $priority = $request->query('priority');
                $statuses = config('support.statuses', []);
                $priorities = config('support.priorities', []);

                if ($search !== '') {
                    $query->where(function ($builder) use ($search) {
                        $builder->where('ticket_number', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('subject', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('requester_name', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('requester_email', 'like', "%" . $this->escapeLike($search) . "%");
                    });
                }

                if ($status) {
                    $query->where('status', $status);
                }

                if ($priority) {
                    $query->where('priority', $priority);
                }

                $this->applyDateRange($query, 'created_at', $startDate, $endDate);

                $fields = [
                    [
                        'name' => 'search',
                        'type' => 'text',
                        'label' => 'Search',
                        'placeholder' => 'Search by ticket, subject, or requester',
                        'value' => $search,
                    ],
                    [
                        'name' => 'status',
                        'type' => 'select',
                        'label' => 'Status',
                        'options' => array_merge(['' => 'All statuses'], $statuses),
                        'value' => $status,
                    ],
                    [
                        'name' => 'priority',
                        'type' => 'select',
                        'label' => 'Priority',
                        'options' => array_merge(['' => 'All priorities'], $priorities),
                        'value' => $priority,
                    ],
                    [
                        'name' => 'start_date',
                        'type' => 'date',
                        'label' => 'From',
                        'value' => $startDate,
                    ],
                    [
                        'name' => 'end_date',
                        'type' => 'date',
                        'label' => 'To',
                        'value' => $endDate,
                    ],
                ];
                break;
            default:
                $fields = [];
        }

        return [
            'fields' => $fields,
        ];
    }

    private function applyDateRange(Builder $query, string $column, ?string $startDate, ?string $endDate): void
    {
        if ($startDate) {
            $query->whereDate($column, '>=', $startDate);
        }

        if ($endDate) {
            $query->whereDate($column, '<=', $endDate);
        }
    }

    private function normalizeDate(?string $value): ?string
    {
        if (! $value) {
            return null;
        }

        try {
            return Carbon::parse($value)->format('Y-m-d');
        } catch (\Throwable $exception) {
            return null;
        }
    }

    private function formatTitle(?string $value): string
    {
        if (! $value) {
            return '-';
        }

        return str_replace('_', ' ', ucwords($value, '_'));
    }
}
