<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\StoreRoleRequest;
use App\Http\Requests\Admin\UpdateRoleRequest;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

class RolesController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', Role::class);

        $search = $request->query('search');

        $query = Role::query()->withCount(['permissions', 'users'])->orderBy('name');

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreRoleRequest $request)
    {
        $this->authorize('create', Role::class);

        $data = $request->validated();
        $permissions = $data['permissions'] ?? [];
        unset($data['permissions']);

        $role = Role::create($data);

        if ($permissions) {
            $permissionIds = $this->filterPermissionsByGuard($permissions, $role->guard_name);
            $role->syncPermissions($permissionIds);
        }

        return $this->success($role->load('permissions'), [], 201);
    }

    public function show(Role $role)
    {
        $this->authorize('view', $role);

        return $this->success($role->load('permissions'));
    }

    public function update(UpdateRoleRequest $request, Role $role)
    {
        $this->authorize('update', $role);

        $data = $request->validated();
        $permissions = $data['permissions'] ?? [];
        unset($data['permissions']);

        $role->update($data);

        if ($permissions) {
            $permissionIds = $this->filterPermissionsByGuard($permissions, $role->guard_name);
            $role->syncPermissions($permissionIds);
        } else {
            $role->syncPermissions([]);
        }

        return $this->success($role->fresh()->load('permissions'));
    }

    public function destroy(Role $role)
    {
        $this->authorize('delete', $role);

        $role->syncPermissions([]);
        $role->delete();

        return $this->success(['message' => 'Role deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', Role::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:roles,id'],
        ]);

        Role::query()->whereIn('id', $validated['ids'])->each(function (Role $role) {
            $role->syncPermissions([]);
            $role->delete();
        });

        return $this->success(['message' => 'Selected roles deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', Role::class);

        $search = $request->query('search');
        $query = Role::query()->with('permissions')->orderBy('name');

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        $columns = [
            ['key' => 'name', 'label' => 'Name'],
            ['key' => 'guard_name', 'label' => 'Guard'],
            ['key' => 'permissions', 'label' => 'Permissions'],
        ];

        $map = fn (Role $role) => [
            'name' => $role->name,
            'guard_name' => $role->guard_name,
            'permissions' => $role->permissions->pluck('name')->implode('|'),
        ];

        $filename = 'roles-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Role Report', 'Role export');
    }

    private function filterPermissionsByGuard(array $permissionIds, string $guard): array
    {
        if (! $permissionIds) {
            return [];
        }

        return Permission::query()
            ->whereIn('id', $permissionIds)
            ->where('guard_name', $guard)
            ->pluck('id')
            ->all();
    }
}
