<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportSecurityAuditLogsRequest;
use App\Http\Requests\Admin\StoreSecurityAuditLogRequest;
use App\Http\Requests\Admin\UpdateSecurityAuditLogRequest;
use App\Models\SecurityAuditLog;
use App\Models\User;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;

class SecurityAuditLogController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', SecurityAuditLog::class);

        $search = $request->query('search');
        $eventType = $request->query('event_type');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = SecurityAuditLog::query()->with('user');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('ip_address', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('event_type', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('status', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhereHas('user', function ($userQuery) use ($search) {
                        $userQuery->where('email', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('name', 'like', "%" . $this->escapeLike($search) . "%");
                    });
            });
        }

        if ($eventType) {
            $query->where('event_type', $eventType);
        }

        if ($status) {
            $query->where('status', $status);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString(), [
            'event_types' => config('security.audit_event_types', []),
            'statuses' => config('security.audit_statuses', []),
        ]);
    }

    public function store(StoreSecurityAuditLogRequest $request)
    {
        $this->authorize('create', SecurityAuditLog::class);

        $data = $request->validated();

        $log = SecurityAuditLog::create([
            'user_id' => $data['user_id'] ?? null,
            'event_type' => $data['event_type'],
            'status' => $data['status'],
            'ip_address' => $data['ip_address'] ?? null,
            'user_agent' => $data['user_agent'] ?? null,
            'metadata' => $this->parseMetadata($data['metadata'] ?? null),
            'logged_at' => $this->parseDateTime($data['logged_at'] ?? null) ?? now(),
        ]);

        return $this->success($log->load('user'), [], 201);
    }

    public function show(SecurityAuditLog $securityAuditLog)
    {
        $this->authorize('view', $securityAuditLog);

        return $this->success($securityAuditLog->load('user'));
    }

    public function update(UpdateSecurityAuditLogRequest $request, SecurityAuditLog $securityAuditLog)
    {
        $this->authorize('update', $securityAuditLog);

        $data = $request->validated();

        $securityAuditLog->update([
            'user_id' => $data['user_id'] ?? null,
            'event_type' => $data['event_type'],
            'status' => $data['status'],
            'ip_address' => $data['ip_address'] ?? null,
            'user_agent' => $data['user_agent'] ?? null,
            'metadata' => $this->parseMetadata($data['metadata'] ?? null),
            'logged_at' => $this->parseDateTime($data['logged_at'] ?? null),
        ]);

        return $this->success($securityAuditLog->fresh()->load('user'));
    }

    public function destroy(SecurityAuditLog $securityAuditLog)
    {
        $this->authorize('delete', $securityAuditLog);

        $securityAuditLog->delete();

        return $this->success(['message' => 'Security audit log deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', SecurityAuditLog::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:security_audit_logs,id'],
        ]);

        SecurityAuditLog::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected security audit logs deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', SecurityAuditLog::class);

        $search = $request->query('search');
        $eventType = $request->query('event_type');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = SecurityAuditLog::query()->with('user');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('ip_address', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('event_type', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('status', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhereHas('user', function ($userQuery) use ($search) {
                        $userQuery->where('email', 'like', "%" . $this->escapeLike($search) . "%")
                            ->orWhere('name', 'like', "%" . $this->escapeLike($search) . "%");
                    });
            });
        }

        if ($eventType) {
            $query->where('event_type', $eventType);
        }

        if ($status) {
            $query->where('status', $status);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'event_type', 'label' => 'Event Type'],
            ['key' => 'status', 'label' => 'Status'],
            ['key' => 'user_email', 'label' => 'User Email'],
            ['key' => 'ip_address', 'label' => 'IP Address'],
            ['key' => 'user_agent', 'label' => 'User Agent'],
            ['key' => 'metadata', 'label' => 'Metadata'],
            ['key' => 'logged_at', 'label' => 'Logged At'],
        ];

        $map = fn (SecurityAuditLog $log) => [
            'event_type' => $log->event_type,
            'status' => $log->status,
            'user_email' => $log->user?->email,
            'ip_address' => $log->ip_address,
            'user_agent' => $log->user_agent,
            'metadata' => $log->metadata ? json_encode($log->metadata) : '',
            'logged_at' => optional($log->logged_at)->format('Y-m-d H:i:s'),
        ];

        $filename = 'security-audit-logs-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Security Audit Logs', 'Security audit export');
    }

    public function import(ImportSecurityAuditLogsRequest $request)
    {
        $this->authorize('import', SecurityAuditLog::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, ['event_type', 'status', 'user_email', 'ip_address', 'user_agent', 'metadata', 'logged_at']);
        $eventTypes = array_keys(config('security.audit_event_types', []));
        $statuses = array_keys(config('security.audit_statuses', []));

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $eventType = $this->normalizeValue($data['event_type'] ?? '', $eventTypes);
            $status = $this->normalizeValue($data['status'] ?? '', $statuses);

            if (! $eventType || ! $status) {
                $skipped++;
                continue;
            }

            $userId = null;
            if (! empty($data['user_email'])) {
                $userId = User::query()->where('email', trim((string) $data['user_email']))->value('id');
            }

            SecurityAuditLog::create([
                'user_id' => $userId,
                'event_type' => $eventType,
                'status' => $status,
                'ip_address' => $data['ip_address'] ?? null,
                'user_agent' => $data['user_agent'] ?? null,
                'metadata' => $this->parseMetadata($data['metadata'] ?? null),
                'logged_at' => $this->parseDateTime($data['logged_at'] ?? null) ?? now(),
            ]);

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function parseDateTime(?string $value): ?Carbon
    {
        if (! $value) {
            return null;
        }

        try {
            return Carbon::parse($value);
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function parseMetadata(?string $value): ?array
    {
        if (! $value) {
            return null;
        }

        $value = trim($value);
        if ($value === '') {
            return null;
        }

        $decoded = json_decode($value, true);
        if (json_last_error() === JSON_ERROR_NONE) {
            return $decoded;
        }

        return ['note' => $value];
    }

    private function normalizeValue(?string $value, array $allowed): ?string
    {
        $value = Str::lower(trim((string) $value));

        return in_array($value, $allowed, true) ? $value : null;
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'event_type_asc' => $query->orderBy('event_type'),
            'event_type_desc' => $query->orderBy('event_type', 'desc'),
            'oldest' => $query->orderBy('logged_at', 'asc'),
            default => $query->orderBy('logged_at', 'desc'),
        };
    }
}
