<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportStatesRequest;
use App\Http\Requests\Admin\StoreStateRequest;
use App\Http\Requests\Admin\UpdateStateRequest;
use App\Models\State;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class StateController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', State::class);

        $search = $request->query('search');
        $countryId = $request->query('country_id');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = State::query()->with('country');

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($countryId) {
            $query->where('country_id', $countryId);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreStateRequest $request)
    {
        $this->authorize('create', State::class);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $state = State::create($data);

        return $this->success($state->load('country'), [], 201);
    }

    public function show(State $state)
    {
        $this->authorize('view', $state);

        return $this->success($state->load('country'));
    }

    public function update(UpdateStateRequest $request, State $state)
    {
        $this->authorize('update', $state);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        $state->update($data);

        return $this->success($state->fresh()->load('country'));
    }

    public function destroy(State $state)
    {
        $this->authorize('delete', $state);

        $state->delete();

        return $this->success(['message' => 'State deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', State::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:states,id'],
        ]);

        State::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected states deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', State::class);

        $search = $request->query('search');
        $countryId = $request->query('country_id');
        $status = $request->query('status');
        $sort = $request->query('sort', 'name_asc');

        $query = State::query()->with('country');

        if ($search) {
            $query->where('name', 'like', "%" . $this->escapeLike($search) . "%");
        }

        if ($countryId) {
            $query->where('country_id', $countryId);
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'name', 'label' => 'Name'],
            ['key' => 'country', 'label' => 'Country'],
            ['key' => 'code', 'label' => 'Code'],
            ['key' => 'is_active', 'label' => 'Active'],
        ];

        $map = fn (State $state) => [
            'name' => $state->name,
            'country' => $state->country?->name ?? '-',
            'code' => $state->code,
            'is_active' => $state->is_active ? 'Yes' : 'No',
        ];

        $filename = 'states-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'States', 'State export');
    }

    public function import(ImportStatesRequest $request)
    {
        $this->authorize('import', State::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, ['country_id', 'country', 'iso2', 'name', 'code', 'is_active']);
        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['name'] ?? ''));

            $countryId = $this->resolveCountryId($data);

            if ($name === '' || ! $countryId) {
                $skipped++;
                continue;
            }

            State::updateOrCreate(
                ['country_id' => $countryId, 'name' => $name],
                [
                    'code' => trim((string) ($data['code'] ?? '')) ?: null,
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_desc' => $query->orderBy('name', 'desc'),
            'latest' => $query->orderBy('created_at', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('name'),
        };
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function resolveCountryId(array $data): ?int
    {
        if (! empty($data['country_id'])) {
            return (int) $data['country_id'];
        }

        $iso2 = strtoupper(trim((string) ($data['iso2'] ?? '')));
        if ($iso2 !== '') {
            return (int) \App\Models\Country::query()->where('iso2', $iso2)->value('id');
        }

        $countryName = trim((string) ($data['country'] ?? ''));
        if ($countryName !== '') {
            return (int) \App\Models\Country::query()->where('name', $countryName)->value('id');
        }

        return null;
    }
}
