<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportSupportTicketsRequest;
use App\Http\Requests\Admin\StoreSupportTicketRequest;
use App\Http\Requests\Admin\UpdateSupportTicketRequest;
use App\Models\SupportTicket;
use App\Models\User;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;

class SupportTicketController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', SupportTicket::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $priority = $request->query('priority');
        $assignee = $request->query('assignee');
        $sort = $request->query('sort', 'latest');

        $query = SupportTicket::query()->with(['currentAssignment.assignedTo']);

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('ticket_number', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('subject', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('requester_name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('requester_email', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status) {
            $query->where('status', $status);
        }

        if ($priority) {
            $query->where('priority', $priority);
        }

        if ($assignee) {
            $query->whereHas('currentAssignment', function ($builder) use ($assignee) {
                $builder->where('assigned_to', $assignee);
            });
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreSupportTicketRequest $request)
    {
        $this->authorize('create', SupportTicket::class);

        $data = $request->validated();

        $ticket = SupportTicket::create([
            'ticket_number' => $this->uniqueTicketNumber(),
            'subject' => $data['subject'],
            'message' => $data['message'],
            'requester_name' => $data['requester_name'],
            'requester_email' => $data['requester_email'],
            'requester_phone' => $data['requester_phone'] ?? null,
            'priority' => $data['priority'],
            'status' => $data['status'],
            'is_active' => $request->boolean('is_active'),
            'resolved_at' => $this->resolveTimestamp($data['status']),
            'created_by' => $request->user()?->id,
        ]);

        return $this->success($ticket, [], 201);
    }

    public function show(SupportTicket $supportTicket)
    {
        $this->authorize('view', $supportTicket);

        $supportTicket->load([
            'assignments.assignedTo',
            'assignments.assignedBy',
            'currentAssignment.assignedTo',
        ]);

        return $this->success($supportTicket);
    }

    public function update(UpdateSupportTicketRequest $request, SupportTicket $supportTicket)
    {
        $this->authorize('update', $supportTicket);

        $data = $request->validated();

        $supportTicket->update([
            'subject' => $data['subject'],
            'message' => $data['message'],
            'requester_name' => $data['requester_name'],
            'requester_email' => $data['requester_email'],
            'requester_phone' => $data['requester_phone'] ?? null,
            'priority' => $data['priority'],
            'status' => $data['status'],
            'is_active' => $request->boolean('is_active'),
            'resolved_at' => $this->resolveTimestamp($data['status'], $supportTicket->resolved_at),
        ]);

        return $this->success($supportTicket->fresh());
    }

    public function destroy(SupportTicket $supportTicket)
    {
        $this->authorize('delete', $supportTicket);

        $supportTicket->assignments()->delete();
        $supportTicket->delete();

        return $this->success(['message' => 'Ticket deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', SupportTicket::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:support_tickets,id'],
        ]);

        $tickets = SupportTicket::query()->whereIn('id', $validated['ids'])->get();
        foreach ($tickets as $ticket) {
            $ticket->assignments()->delete();
            $ticket->delete();
        }

        return $this->success(['message' => 'Selected tickets deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', SupportTicket::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $priority = $request->query('priority');
        $assignee = $request->query('assignee');
        $sort = $request->query('sort', 'latest');

        $query = SupportTicket::query()->with('currentAssignment');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('ticket_number', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('subject', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('requester_name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('requester_email', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status) {
            $query->where('status', $status);
        }

        if ($priority) {
            $query->where('priority', $priority);
        }

        if ($assignee) {
            $query->whereHas('currentAssignment', function ($builder) use ($assignee) {
                $builder->where('assigned_to', $assignee);
            });
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'ticket_number', 'label' => 'Ticket'],
            ['key' => 'subject', 'label' => 'Subject'],
            ['key' => 'priority', 'label' => 'Priority'],
            ['key' => 'status', 'label' => 'Status'],
            ['key' => 'requester', 'label' => 'Requester'],
            ['key' => 'requester_email', 'label' => 'Email'],
            ['key' => 'assigned_to', 'label' => 'Assigned To'],
            ['key' => 'resolved_at', 'label' => 'Resolved'],
        ];

        $map = fn (SupportTicket $ticket) => [
            'ticket_number' => $ticket->ticket_number,
            'subject' => $ticket->subject,
            'priority' => $ticket->priority,
            'status' => $ticket->status,
            'requester' => $ticket->requester_name,
            'requester_email' => $ticket->requester_email,
            'assigned_to' => $ticket->currentAssignment?->assigned_to,
            'resolved_at' => optional($ticket->resolved_at)->format('Y-m-d H:i:s'),
        ];

        $filename = 'support-tickets-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Support Tickets', 'Support ticket export');
    }

    public function import(ImportSupportTicketsRequest $request)
    {
        $this->authorize('import', SupportTicket::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, [
            'ticket_number',
            'subject',
            'message',
            'requester_name',
            'requester_email',
            'requester_phone',
            'priority',
            'status',
            'is_active',
        ]);

        $imported = 0;
        $skipped = 0;
        $priorities = array_keys(config('support.priorities', []));
        $statuses = array_keys(config('support.statuses', []));

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $subject = trim((string) ($data['subject'] ?? ''));
            $message = trim((string) ($data['message'] ?? ''));
            $requesterName = trim((string) ($data['requester_name'] ?? ''));
            $requesterEmail = trim((string) ($data['requester_email'] ?? ''));
            $priority = $this->normalizeValue($data['priority'] ?? null, $priorities, 'medium');
            $status = $this->normalizeValue($data['status'] ?? null, $statuses, 'open');

            if ($subject === '' || $message === '' || $requesterName === '' || $requesterEmail === '') {
                $skipped++;
                continue;
            }

            $ticketNumber = trim((string) ($data['ticket_number'] ?? ''));
            if ($ticketNumber === '') {
                $ticketNumber = $this->uniqueTicketNumber();
            }

            SupportTicket::updateOrCreate(
                ['ticket_number' => $ticketNumber],
                [
                    'subject' => $subject,
                    'message' => $message,
                    'requester_name' => $requesterName,
                    'requester_email' => $requesterEmail,
                    'requester_phone' => $data['requester_phone'] ?? null,
                    'priority' => $priority,
                    'status' => $status,
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                    'resolved_at' => $this->resolveTimestamp($status),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function uniqueTicketNumber(): string
    {
        do {
            $number = 'TKT-' . now()->format('Ymd') . '-' . strtoupper(Str::random(6));
        } while (SupportTicket::query()->where('ticket_number', $number)->exists());

        return $number;
    }

    private function resolveTimestamp(string $status, ?Carbon $current = null): ?Carbon
    {
        if (in_array($status, ['resolved', 'closed'], true)) {
            return $current ?? now();
        }

        return null;
    }

    private function normalizeValue(?string $value, array $allowed, string $fallback): string
    {
        if (! $value) {
            return $fallback;
        }

        $value = Str::lower(trim($value));

        return in_array($value, $allowed, true) ? $value : $fallback;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'oldest' => $query->orderBy('created_at', 'asc'),
            'subject_asc' => $query->orderBy('subject'),
            'subject_desc' => $query->orderBy('subject', 'desc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }
}
