<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportSystemLogsRequest;
use App\Http\Requests\Admin\StoreSystemLogRequest;
use App\Http\Requests\Admin\UpdateSystemLogRequest;
use App\Models\SystemLog;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;

class SystemLogController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', SystemLog::class);

        $search = $request->query('search');
        $level = $request->query('level');
        $sort = $request->query('sort', 'latest');

        $query = SystemLog::query()->with('createdBy');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('message', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('source', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($level) {
            $query->where('level', $level);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString(), [
            'levels' => config('system.log_levels', []),
        ]);
    }

    public function store(StoreSystemLogRequest $request)
    {
        $this->authorize('create', SystemLog::class);

        $data = $request->validated();

        $log = SystemLog::create([
            'level' => $data['level'],
            'source' => $data['source'] ?? null,
            'message' => $data['message'],
            'context' => $this->parseContext($data['context'] ?? null),
            'logged_at' => $this->parseDateTime($data['logged_at'] ?? null) ?? now(),
            'created_by' => $request->user()?->id,
        ]);

        return $this->success($log->load('createdBy'), [], 201);
    }

    public function show(SystemLog $systemLog)
    {
        $this->authorize('view', $systemLog);

        return $this->success($systemLog->load('createdBy'));
    }

    public function update(UpdateSystemLogRequest $request, SystemLog $systemLog)
    {
        $this->authorize('update', $systemLog);

        $data = $request->validated();

        $systemLog->update([
            'level' => $data['level'],
            'source' => $data['source'] ?? null,
            'message' => $data['message'],
            'context' => $this->parseContext($data['context'] ?? null),
            'logged_at' => $this->parseDateTime($data['logged_at'] ?? null),
        ]);

        return $this->success($systemLog->fresh()->load('createdBy'));
    }

    public function destroy(SystemLog $systemLog)
    {
        $this->authorize('delete', $systemLog);

        $systemLog->delete();

        return $this->success(['message' => 'System log deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', SystemLog::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:system_logs,id'],
        ]);

        SystemLog::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected system logs deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', SystemLog::class);

        $search = $request->query('search');
        $level = $request->query('level');
        $sort = $request->query('sort', 'latest');

        $query = SystemLog::query()->with('createdBy');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('message', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('source', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($level) {
            $query->where('level', $level);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'level', 'label' => 'Level'],
            ['key' => 'source', 'label' => 'Source'],
            ['key' => 'message', 'label' => 'Message'],
            ['key' => 'context', 'label' => 'Context'],
            ['key' => 'logged_at', 'label' => 'Logged At'],
            ['key' => 'created_by', 'label' => 'Created By'],
        ];

        $map = fn (SystemLog $log) => [
            'level' => $log->level,
            'source' => $log->source,
            'message' => $log->message,
            'context' => $log->context ? json_encode($log->context) : '',
            'logged_at' => optional($log->logged_at)->format('Y-m-d H:i:s'),
            'created_by' => $log->createdBy?->email,
        ];

        $filename = 'system-logs-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'System Logs', 'System log export');
    }

    public function import(ImportSystemLogsRequest $request)
    {
        $this->authorize('import', SystemLog::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, ['level', 'source', 'message', 'context', 'logged_at']);
        $levels = array_keys(config('system.log_levels', []));

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $message = trim((string) ($data['message'] ?? ''));

            if ($message === '') {
                $skipped++;
                continue;
            }

            $level = $this->normalizeLevel($data['level'] ?? null, $levels);

            SystemLog::create([
                'level' => $level,
                'source' => $data['source'] ?? null,
                'message' => $message,
                'context' => $this->parseContext($data['context'] ?? null),
                'logged_at' => $this->parseDateTime($data['logged_at'] ?? null) ?? now(),
            ]);

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function parseDateTime(?string $value): ?Carbon
    {
        if (! $value) {
            return null;
        }

        try {
            return Carbon::parse($value);
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function parseContext(?string $value): ?array
    {
        if (! $value) {
            return null;
        }

        $value = trim($value);
        if ($value === '') {
            return null;
        }

        $decoded = json_decode($value, true);
        if (json_last_error() === JSON_ERROR_NONE) {
            return $decoded;
        }

        return ['note' => $value];
    }

    private function normalizeLevel(?string $value, array $levels): string
    {
        $value = Str::lower(trim((string) $value));

        return in_array($value, $levels, true) ? $value : 'info';
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'oldest' => $query->orderBy('logged_at', 'asc'),
            'level_asc' => $query->orderBy('level'),
            'level_desc' => $query->orderBy('level', 'desc'),
            default => $query->orderBy('logged_at', 'desc'),
        };
    }
}
