<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\UpdateSystemSettingsRequest;
use App\Models\City;
use App\Models\Country;
use App\Models\Currency;
use App\Models\Language;
use App\Models\State;
use App\Models\SystemSetting;
use App\Support\FormatSettings;
use App\Support\SeoFileManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;

class SystemSettingsController extends ApiController
{
    public function show()
    {
        $this->authorize('viewAny', SystemSetting::class);

        $setting = SystemSetting::query()->first();
        $payload = $setting?->payload ?? [];

        $socialProviders = config('system.social_providers', []);
        $defaultProviders = [];
        foreach ($socialProviders as $key => $label) {
            $defaultProviders[$key] = [
                'enabled' => false,
                'client_id' => '',
                'client_secret' => '',
                'redirect_url' => '',
            ];
        }

        $defaults = [
            'general' => [
                'default_language' => '',
                'default_currency' => '',
                'default_timezone' => '',
                'date_format' => '',
                'time_format' => '',
                'support_email' => '',
                'support_phone' => '',
                'enable_multi_language' => false,
            ],
            'site' => [
                'site_name' => '',
                'site_logo' => null,
                'favicon' => null,
                'admin_logo' => null,
                'company_name' => '',
                'company_address' => '',
                'company_country_id' => null,
                'company_state_id' => null,
                'company_city_id' => null,
                'company_phone' => '',
                'business_hours' => '',
                'latitude' => '',
                'longitude' => '',
                'right_click_disable' => false,
                'uc_browser_block' => false,
            ],
            'smtp' => [
                'smtp_host' => '',
                'smtp_port' => '',
                'smtp_username' => '',
                'smtp_password' => '',
                'smtp_encryption' => 'tls',
                'smtp_from_address' => '',
                'smtp_from_name' => '',
            ],
            'footer' => [
                'footer_logo' => null,
                'footer_social' => [
                    'facebook' => '',
                    'x' => '',
                    'linkedin' => '',
                    'instagram' => '',
                    'whatsapp' => '',
                ],
                'app_links' => [
                    'android' => '',
                    'ios' => '',
                    'direct' => '',
                ],
            ],
            'seo' => [
                'meta_title' => '',
                'meta_description' => '',
                'meta_keywords' => '',
                'sitemap_enabled' => false,
                'robots_txt' => '',
                'llm_txt' => '',
            ],
            'social' => [
                'providers' => $defaultProviders,
            ],
            'gdpr' => [
                'gdpr_enabled' => false,
                'gdpr_message' => '',
                'gdpr_accept_label' => 'Accept',
                'gdpr_decline_label' => 'Decline',
            ],
            'media' => [
                'max_file_size' => (int) config('events.media.max_file_size', 51200),
                'chunk_size' => (int) config('events.media.chunk_size', 5242880),
                'max_width' => (int) config('events.media.max_width', 2000),
                'thumbnail_width' => (int) config('events.media.thumbnail_width', 600),
                'quality' => (int) config('events.media.quality', 82),
                'storage_limit_gb' => (float) config('events.media.storage_limit_gb', 6.1),
                'watermark' => [
                    'enabled' => (bool) config('events.media.watermark.enabled', true),
                    'image' => (string) config('events.media.watermark.image', ''),
                    'position' => (string) config('events.media.watermark.position', 'top_right'),
                ],
                'ftp_path' => (string) config('events.media.ftp_path', storage_path('app/ftp')),
            ],
            'maintenance' => [
                'maintenance_enabled' => false,
                'maintenance_message' => '',
            ],
        ];

        $settings = array_replace_recursive($defaults, $payload);

        return $this->success([
            'settings' => $settings,
            'social_providers' => $socialProviders,
            'countries' => Country::query()->orderBy('name')->get(['id', 'name']),
            'states' => State::query()->orderBy('name')->get(['id', 'name', 'country_id']),
            'cities' => City::query()->orderBy('name')->get(['id', 'name', 'state_id']),
            'currencies' => Currency::query()->orderBy('name')->get(['id', 'name', 'code']),
            'languages' => Language::query()->orderBy('name')->get(['id', 'name', 'code']),
        ]);
    }

    public function update(UpdateSystemSettingsRequest $request, string $group)
    {
        $this->authorize('update', SystemSetting::class);

        $setting = SystemSetting::query()->firstOrCreate([]);
        $payload = $setting->payload ?? [];
        $data = $request->validated();

        switch ($group) {
            case 'general':
                $data['enable_multi_language'] = $request->boolean('enable_multi_language');
                break;
            case 'site':
                $data['right_click_disable'] = $request->boolean('right_click_disable');
                $data['uc_browser_block'] = $request->boolean('uc_browser_block');
                $data = $this->handleUpload($request, $data, 'site_logo', 'settings/site');
                $data = $this->handleUpload($request, $data, 'favicon', 'settings/site');
                $data = $this->handleUpload($request, $data, 'admin_logo', 'settings/site');
                break;
            case 'smtp':
                break;
            case 'footer':
                $data = $this->handleUpload($request, $data, 'footer_logo', 'settings/footer');
                $data['footer_social'] = $this->normalizeLinks(
                    $data['footer_social'] ?? [],
                    ['facebook', 'x', 'linkedin', 'instagram', 'whatsapp']
                );
                $data['app_links'] = $this->normalizeLinks(
                    $data['app_links'] ?? [],
                    ['android', 'ios', 'direct']
                );
                break;
            case 'seo':
                $data['sitemap_enabled'] = $request->boolean('sitemap_enabled');
                break;
            case 'social':
                $data['providers'] = $this->normalizeProviders($data['providers'] ?? []);
                break;
            case 'gdpr':
                $data['gdpr_enabled'] = $request->boolean('gdpr_enabled');
                break;
            case 'media':
                $maxFileSize = $data['media_max_file_size'] ?? $data['max_file_size'] ?? null;
                $data['max_file_size'] = $this->convertMbToKb($maxFileSize);
                $data['chunk_size'] = $this->convertMbToBytes($data['chunk_size'] ?? null);
                $data['max_width'] = isset($data['max_width']) ? (int) $data['max_width'] : null;
                $data['thumbnail_width'] = isset($data['thumbnail_width']) ? (int) $data['thumbnail_width'] : null;
                $data['quality'] = isset($data['quality']) ? (int) $data['quality'] : null;
                $data['storage_limit_gb'] = isset($data['storage_limit_gb']) ? (float) $data['storage_limit_gb'] : null;
                $data['ftp_path'] = isset($data['ftp_path']) ? trim((string) $data['ftp_path']) : null;
                $data = $this->handleUpload($request, $data, 'watermark_image', 'settings/media');
                $watermark = is_array($payload['media']['watermark'] ?? null) ? $payload['media']['watermark'] : [];
                $watermarkImage = $data['watermark_image'] ?? ($watermark['image'] ?? null);
                $watermarkPosition = trim((string) ($data['watermark_position'] ?? ''));
                if ($watermarkPosition === '') {
                    $watermarkPosition = (string) config('events.media.watermark.position', 'top_right');
                }
                $data['watermark'] = [
                    'enabled' => $request->boolean('watermark_enabled'),
                    'image' => $watermarkImage,
                    'position' => $watermarkPosition,
                ];
                unset($data['media_max_file_size'], $data['watermark_enabled'], $data['watermark_image'], $data['watermark_position']);
                break;
            case 'maintenance':
                $data['maintenance_enabled'] = $request->boolean('maintenance_enabled');
                break;
        }

        $payload[$group] = array_merge($payload[$group] ?? [], $data);
        $setting->payload = $payload;
        $setting->updated_by = $request->user()?->id;
        $setting->save();

        if ($group === 'general') {
            FormatSettings::reset();
        }
        if ($group === 'seo') {
            SeoFileManager::sync($payload['seo'] ?? []);
        }
        if ($group === 'media') {
            \App\Support\EventMediaSettings::forgetCache();
        }

        return $this->success([
            'message' => 'Settings updated.',
            'settings' => $setting->payload,
        ]);
    }

    public function testSmtp(Request $request)
    {
        $this->authorize('update', SystemSetting::class);

        $data = $request->validate([
            'test_email' => ['required', 'email', 'max:255'],
        ]);

        $recipient = $data['test_email'];
        $appName = config('app.name', 'SnapNest');

        try {
            Mail::raw("SMTP test email from {$appName}.", function ($message) use ($recipient, $appName) {
                $message->to($recipient)
                    ->subject("{$appName} SMTP Test");
            });
        } catch (\Throwable $exception) {
            return $this->success([
                'message' => 'SMTP test failed: ' . $exception->getMessage(),
            ], [], 422);
        }

        return $this->success(['message' => 'SMTP test email sent.']);
    }

    private function handleUpload(Request $request, array $data, string $field, string $path): array
    {
        if ($request->hasFile($field)) {
            $data[$field] = $request->file($field)->store($path, 'public');
        }

        return $data;
    }

    private function normalizeLinks(array $links, array $keys): array
    {
        $normalized = [];

        foreach ($keys as $key) {
            $normalized[$key] = isset($links[$key]) ? trim((string) $links[$key]) : '';
        }

        return $normalized;
    }

    private function normalizeProviders(array $providers): array
    {
        $defaults = [];
        $available = config('system.social_providers', []);

        foreach ($available as $key => $label) {
            $data = $providers[$key] ?? [];
            $defaults[$key] = [
                'enabled' => filter_var($data['enabled'] ?? false, FILTER_VALIDATE_BOOLEAN),
                'client_id' => trim((string) ($data['client_id'] ?? '')),
                'client_secret' => trim((string) ($data['client_secret'] ?? '')),
                'redirect_url' => trim((string) ($data['redirect_url'] ?? '')),
            ];
        }

        return $defaults;
    }

    private function convertMbToKb($value): ?int
    {
        if ($value === null || trim((string) $value) === '') {
            return null;
        }

        return (int) round(((float) $value) * 1024);
    }

    private function convertMbToBytes($value): ?int
    {
        if ($value === null || trim((string) $value) === '') {
            return null;
        }

        return (int) round(((float) $value) * 1024 * 1024);
    }
}
