<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportTestimonialsRequest;
use App\Http\Requests\Admin\StoreTestimonialRequest;
use App\Http\Requests\Admin\UpdateTestimonialRequest;
use App\Models\Testimonial;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class TestimonialController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', Testimonial::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = Testimonial::query()->with('creator');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('client_name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('client_position', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('review', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreTestimonialRequest $request)
    {
        $this->authorize('create', Testimonial::class);

        $data = $request->validated();

        if ($request->hasFile('client_image')) {
            $data['client_image'] = $request->file('client_image')->store('content/testimonials', 'public');
        }

        $data['is_active'] = $request->boolean('is_active');
        $data['created_by'] = $request->user()?->id;

        $testimonial = Testimonial::create($data);

        return $this->success($testimonial->load('creator'), [], 201);
    }

    public function show(Testimonial $testimonial)
    {
        $this->authorize('view', $testimonial);

        return $this->success($testimonial->load('creator'));
    }

    public function update(UpdateTestimonialRequest $request, Testimonial $testimonial)
    {
        $this->authorize('update', $testimonial);

        $data = $request->validated();
        $data['is_active'] = $request->boolean('is_active');

        if ($request->hasFile('client_image')) {
            if ($testimonial->client_image) {
                Storage::disk('public')->delete($testimonial->client_image);
            }
            $data['client_image'] = $request->file('client_image')->store('content/testimonials', 'public');
        }

        $testimonial->update($data);

        return $this->success($testimonial->fresh()->load('creator'));
    }

    public function destroy(Testimonial $testimonial)
    {
        $this->authorize('delete', $testimonial);

        if ($testimonial->client_image) {
            Storage::disk('public')->delete($testimonial->client_image);
        }

        $testimonial->delete();

        return $this->success(['message' => 'Testimonial deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', Testimonial::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:testimonials,id'],
        ]);

        $testimonials = Testimonial::query()->whereIn('id', $validated['ids'])->get();

        foreach ($testimonials as $testimonial) {
            if ($testimonial->client_image) {
                Storage::disk('public')->delete($testimonial->client_image);
            }
            $testimonial->delete();
        }

        return $this->success(['message' => 'Selected testimonials deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', Testimonial::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $sort = $request->query('sort', 'latest');

        $query = Testimonial::query()->with('creator');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('client_name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('client_position', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('review', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'client_name', 'label' => 'Client Name'],
            ['key' => 'client_position', 'label' => 'Client Position'],
            ['key' => 'rating', 'label' => 'Rating'],
            ['key' => 'review', 'label' => 'Review'],
            ['key' => 'is_active', 'label' => 'Status'],
            ['key' => 'created_by', 'label' => 'Created By'],
            ['key' => 'created_at', 'label' => 'Created At'],
        ];

        $map = fn (Testimonial $testimonial) => [
            'client_name' => $testimonial->client_name,
            'client_position' => $testimonial->client_position,
            'rating' => $testimonial->rating,
            'review' => $testimonial->review,
            'is_active' => $testimonial->is_active ? 'Active' : 'Inactive',
            'created_by' => $testimonial->creator?->email,
            'created_at' => optional($testimonial->created_at)->format('Y-m-d H:i:s'),
        ];

        $filename = 'testimonials-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Testimonials Report', 'Testimonials export');
    }

    public function import(ImportTestimonialsRequest $request)
    {
        $this->authorize('create', Testimonial::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, ['client_name', 'client_position', 'rating', 'review', 'is_active']);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $name = trim((string) ($data['client_name'] ?? ''));
            $review = trim((string) ($data['review'] ?? ''));

            if ($name === '' || $review === '') {
                $skipped++;
                continue;
            }

            $position = trim((string) ($data['client_position'] ?? ''));
            $rating = $this->normalizeRating($data['rating'] ?? null);

            Testimonial::updateOrCreate(
                [
                    'client_name' => $name,
                    'client_position' => $position !== '' ? $position : null,
                ],
                [
                    'review' => $review,
                    'rating' => $rating,
                    'is_active' => $this->parseBool($data['is_active'] ?? null, true),
                ]
            );

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'message' => "Imported {$imported} testimonials. Skipped {$skipped} rows.",
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function normalizeRating($value): int
    {
        $rating = (int) $value;

        if ($rating < 1) {
            return 1;
        }

        if ($rating > 5) {
            return 5;
        }

        return $rating ?: 5;
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim($value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'name_asc' => $query->orderBy('client_name'),
            'name_desc' => $query->orderBy('client_name', 'desc'),
            'rating_asc' => $query->orderBy('rating'),
            'rating_desc' => $query->orderBy('rating', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }
}
