<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\ImportTicketAssignmentsRequest;
use App\Http\Requests\Admin\StoreTicketAssignmentRequest;
use App\Http\Requests\Admin\UpdateTicketAssignmentRequest;
use App\Models\SupportTicket;
use App\Models\SupportTicketAssignment;
use App\Models\User;
use App\Services\ExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class TicketAssignmentController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', SupportTicketAssignment::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $ticketFilter = $request->query('ticket');
        $assignee = $request->query('assignee');
        $sort = $request->query('sort', 'latest');

        $query = SupportTicketAssignment::query()
            ->with(['ticket', 'assignedTo', 'assignedBy']);

        if ($search) {
            $query->whereHas('ticket', function ($builder) use ($search) {
                $builder->where('ticket_number', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('subject', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        if ($ticketFilter) {
            $query->where('support_ticket_id', $ticketFilter);
        }

        if ($assignee) {
            $query->where('assigned_to', $assignee);
        }

        $this->applySort($query, $sort);

        return $this->paginated($query->paginate(15)->withQueryString(), [
            'tickets' => SupportTicket::query()->orderBy('created_at', 'desc')->limit(200)->get(['id', 'ticket_number', 'subject']),
            'agents' => User::role('Support Agent')->orderBy('name')->get(['id', 'name', 'email']),
        ]);
    }

    public function store(StoreTicketAssignmentRequest $request)
    {
        $this->authorize('create', SupportTicketAssignment::class);

        $data = $request->validated();

        $assignment = null;

        DB::transaction(function () use ($data, $request, &$assignment) {
            if ($request->boolean('is_active')) {
                SupportTicketAssignment::query()
                    ->where('support_ticket_id', $data['support_ticket_id'])
                    ->update(['is_active' => false]);
            }

            $assignment = SupportTicketAssignment::create([
                'support_ticket_id' => $data['support_ticket_id'],
                'assigned_to' => $data['assigned_to'],
                'assigned_by' => $request->user()?->id,
                'note' => $data['note'] ?? null,
                'is_active' => $request->boolean('is_active'),
                'assigned_at' => $data['assigned_at'] ?? now(),
            ]);
        });

        return $this->success($assignment?->load(['ticket', 'assignedTo', 'assignedBy']), [], 201);
    }

    public function show(SupportTicketAssignment $ticketAssignment)
    {
        $this->authorize('view', $ticketAssignment);

        return $this->success($ticketAssignment->load(['ticket', 'assignedTo', 'assignedBy']));
    }

    public function update(UpdateTicketAssignmentRequest $request, SupportTicketAssignment $ticketAssignment)
    {
        $this->authorize('update', $ticketAssignment);

        $data = $request->validated();

        DB::transaction(function () use ($data, $request, $ticketAssignment) {
            if ($request->boolean('is_active')) {
                SupportTicketAssignment::query()
                    ->where('support_ticket_id', $data['support_ticket_id'])
                    ->where('id', '!=', $ticketAssignment->id)
                    ->update(['is_active' => false]);
            }

            $ticketAssignment->update([
                'support_ticket_id' => $data['support_ticket_id'],
                'assigned_to' => $data['assigned_to'],
                'note' => $data['note'] ?? null,
                'is_active' => $request->boolean('is_active'),
                'assigned_at' => $data['assigned_at'] ?? $ticketAssignment->assigned_at,
            ]);
        });

        return $this->success($ticketAssignment->fresh()->load(['ticket', 'assignedTo', 'assignedBy']));
    }

    public function destroy(SupportTicketAssignment $ticketAssignment)
    {
        $this->authorize('delete', $ticketAssignment);

        $ticketAssignment->delete();

        return $this->success(['message' => 'Assignment deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', SupportTicketAssignment::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:support_ticket_assignments,id'],
        ]);

        SupportTicketAssignment::query()->whereIn('id', $validated['ids'])->delete();

        return $this->success(['message' => 'Selected assignments deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', SupportTicketAssignment::class);

        $search = $request->query('search');
        $status = $request->query('status');
        $ticketFilter = $request->query('ticket');
        $assignee = $request->query('assignee');
        $sort = $request->query('sort', 'latest');

        $query = SupportTicketAssignment::query()->with(['ticket', 'assignedTo']);

        if ($search) {
            $query->whereHas('ticket', function ($builder) use ($search) {
                $builder->where('ticket_number', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('subject', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($status === 'active') {
            $query->where('is_active', true);
        } elseif ($status === 'inactive') {
            $query->where('is_active', false);
        }

        if ($ticketFilter) {
            $query->where('support_ticket_id', $ticketFilter);
        }

        if ($assignee) {
            $query->where('assigned_to', $assignee);
        }

        $this->applySort($query, $sort);

        $columns = [
            ['key' => 'ticket_number', 'label' => 'Ticket'],
            ['key' => 'subject', 'label' => 'Subject'],
            ['key' => 'assigned_to', 'label' => 'Assigned To'],
            ['key' => 'assigned_to_email', 'label' => 'Assignee Email'],
            ['key' => 'assigned_at', 'label' => 'Assigned At'],
            ['key' => 'is_active', 'label' => 'Status'],
            ['key' => 'note', 'label' => 'Note'],
        ];

        $map = fn (SupportTicketAssignment $assignment) => [
            'ticket_number' => $assignment->ticket?->ticket_number,
            'subject' => $assignment->ticket?->subject,
            'assigned_to' => $assignment->assignedTo?->name,
            'assigned_to_email' => $assignment->assignedTo?->email,
            'assigned_at' => optional($assignment->assigned_at)->format('Y-m-d H:i:s'),
            'is_active' => $assignment->is_active ? 'Active' : 'Inactive',
            'note' => $assignment->note,
        ];

        $filename = 'ticket-assignments-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'Ticket Assignments', 'Ticket assignment export');
    }

    public function import(ImportTicketAssignmentsRequest $request)
    {
        $this->authorize('import', SupportTicketAssignment::class);

        $path = $request->file('file')->getRealPath();
        $handle = fopen($path, 'r');

        if (! $handle) {
            return $this->success(['message' => 'Unable to read the uploaded file.'], [], 422);
        }

        $header = fgetcsv($handle);
        if (! $header) {
            fclose($handle);
            return $this->success(['message' => 'CSV file is empty.'], [], 422);
        }

        $map = $this->mapHeaders($header, [
            'ticket_number',
            'assigned_to',
            'assigned_to_email',
            'assigned_at',
            'is_active',
            'note',
        ]);

        $imported = 0;
        $skipped = 0;

        while (($row = fgetcsv($handle)) !== false) {
            if (! array_filter($row)) {
                continue;
            }

            $data = $this->extractRow($row, $map);
            $ticketNumber = trim((string) ($data['ticket_number'] ?? ''));
            $assignedEmail = trim((string) ($data['assigned_to_email'] ?? ''));
            $assignedName = trim((string) ($data['assigned_to'] ?? ''));

            if ($ticketNumber === '') {
                $skipped++;
                continue;
            }

            $ticketId = SupportTicket::query()
                ->where('ticket_number', $ticketNumber)
                ->value('id');

            if (! $ticketId) {
                $skipped++;
                continue;
            }

            $assignedTo = null;
            if ($assignedEmail !== '') {
                $assignedTo = User::query()->where('email', $assignedEmail)->value('id');
            }

            if (! $assignedTo && $assignedName !== '') {
                $assignedTo = User::query()->where('name', $assignedName)->value('id');
            }

            if (! $assignedTo) {
                $skipped++;
                continue;
            }

            $assignedAt = $this->parseDateTime($data['assigned_at'] ?? null) ?? now();
            $isActive = $this->parseBool($data['is_active'] ?? null, true);

            DB::transaction(function () use ($ticketId, $assignedTo, $assignedAt, $isActive, $data) {
                if ($isActive) {
                    SupportTicketAssignment::query()
                        ->where('support_ticket_id', $ticketId)
                        ->update(['is_active' => false]);
                }

                SupportTicketAssignment::create([
                    'support_ticket_id' => $ticketId,
                    'assigned_to' => $assignedTo,
                    'note' => $data['note'] ?? null,
                    'is_active' => $isActive,
                    'assigned_at' => $assignedAt,
                ]);
            });

            $imported++;
        }

        fclose($handle);

        return $this->success([
            'imported' => $imported,
            'skipped' => $skipped,
        ]);
    }

    private function parseDateTime(?string $value): ?Carbon
    {
        if (! $value) {
            return null;
        }

        try {
            return Carbon::parse($value);
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function parseBool(?string $value, bool $default): bool
    {
        if ($value === null || trim((string) $value) === '') {
            return $default;
        }

        $value = Str::lower(trim($value));

        return in_array($value, ['1', 'true', 'yes', 'y'], true);
    }

    private function mapHeaders(array $header, array $allowed): array
    {
        $map = [];

        foreach ($header as $index => $column) {
            $key = Str::slug($column, '_');
            if (in_array($key, $allowed, true)) {
                $map[$index] = $key;
            }
        }

        return $map;
    }

    private function extractRow(array $row, array $map): array
    {
        $data = [];

        foreach ($map as $index => $key) {
            $data[$key] = $row[$index] ?? null;
        }

        return $data;
    }

    private function applySort($query, string $sort): void
    {
        match ($sort) {
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('created_at', 'desc'),
        };
    }
}
