<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\Admin\StoreUserRequest;
use App\Http\Requests\Admin\UpdateUserRequest;
use App\Models\User;
use App\Services\ExportService;
use App\Support\UserDefaults;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class UsersController extends ApiController
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', User::class);

        $search = $request->query('search');
        $roleFilter = $request->query('role');

        $query = User::query()->with(['roles', 'country', 'state', 'city'])->orderBy('name');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('email', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('phone', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($roleFilter) {
            $query->whereHas('roles', function ($builder) use ($roleFilter) {
                $builder->where('name', $roleFilter);
            });
        }

        return $this->paginated($query->paginate(15)->withQueryString());
    }

    public function store(StoreUserRequest $request)
    {
        $this->authorize('create', User::class);

        $data = $request->validated();
        $roles = $data['roles'] ?? [];
        unset($data['roles'], $data['email_verified']);

        $data['password'] = Hash::make($data['password']);
        $data['is_active'] = $request->boolean('is_active', true);
        $data['email_verified_at'] = $request->boolean('email_verified') ? now() : null;
        if (empty($data['plan_id'])) {
            $data['plan_id'] = UserDefaults::defaultPlanId();
        }

        if ($request->hasFile('profile_photo')) {
            $data['profile_photo_path'] = $request->file('profile_photo')->store('avatars', 'public');
        }

        $user = User::create($data);
        $user->syncRoles($roles);
        if (empty($roles)) {
            UserDefaults::assignDefaultRole($user);
        }

        return $this->success($user->load(['roles', 'country', 'state', 'city']), [], 201);
    }

    public function show(User $user)
    {
        $this->authorize('view', $user);

        return $this->success($user->load(['roles', 'country', 'state', 'city']));
    }

    public function update(UpdateUserRequest $request, User $user)
    {
        $this->authorize('update', $user);

        $data = $request->validated();
        $roles = $data['roles'] ?? [];
        unset($data['roles'], $data['email_verified']);

        $data['is_active'] = $request->boolean('is_active', true);
        $data['email_verified_at'] = $request->boolean('email_verified') ? now() : null;

        if (! empty($data['password'])) {
            $data['password'] = Hash::make($data['password']);
        } else {
            unset($data['password']);
        }

        if ($request->hasFile('profile_photo')) {
            if ($user->profile_photo_path) {
                Storage::disk('public')->delete($user->profile_photo_path);
            }
            $data['profile_photo_path'] = $request->file('profile_photo')->store('avatars', 'public');
        }

        $user->update($data);
        $user->syncRoles($roles);

        return $this->success($user->fresh()->load(['roles', 'country', 'state', 'city']));
    }

    public function destroy(User $user)
    {
        $this->authorize('delete', $user);

        if ($user->id === auth()->id()) {
            return $this->success(['message' => 'You cannot delete your own account.'], [], 422);
        }

        if ($user->profile_photo_path) {
            Storage::disk('public')->delete($user->profile_photo_path);
        }

        $user->delete();

        return $this->success(['message' => 'User deleted.']);
    }

    public function bulkDestroy(Request $request)
    {
        $this->authorize('delete', User::class);

        $validated = $request->validate([
            'ids' => ['required', 'array'],
            'ids.*' => ['integer', 'exists:users,id'],
        ]);

        $ids = collect($validated['ids'])->unique()->values();

        if ($ids->contains(auth()->id())) {
            return $this->success(['message' => 'You cannot delete your own account.'], [], 422);
        }

        User::query()->whereIn('id', $ids)->each(function (User $user) {
            if ($user->profile_photo_path) {
                Storage::disk('public')->delete($user->profile_photo_path);
            }
            $user->delete();
        });

        return $this->success(['message' => 'Selected users deleted.']);
    }

    public function export(Request $request, ExportService $exporter)
    {
        $this->authorize('export', User::class);

        $search = $request->query('search');
        $roleFilter = $request->query('role');

        $query = User::query()->with('roles')->orderBy('name');

        if ($search) {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('email', 'like', "%" . $this->escapeLike($search) . "%")
                    ->orWhere('phone', 'like', "%" . $this->escapeLike($search) . "%");
            });
        }

        if ($roleFilter) {
            $query->whereHas('roles', function ($builder) use ($roleFilter) {
                $builder->where('name', $roleFilter);
            });
        }

        $columns = [
            ['key' => 'name', 'label' => 'Name'],
            ['key' => 'email', 'label' => 'Email'],
            ['key' => 'phone', 'label' => 'Phone'],
            ['key' => 'roles', 'label' => 'Roles'],
            ['key' => 'status', 'label' => 'Status'],
            ['key' => 'verified', 'label' => 'Verified'],
            ['key' => 'created_at', 'label' => 'Created'],
        ];

        $map = fn (User $user) => [
            'name' => $user->name,
            'email' => $user->email,
            'phone' => $user->phone ?? '-',
            'roles' => $user->roles->pluck('name')->implode(', '),
            'status' => $user->is_active ? 'Active' : 'Inactive',
            'verified' => $user->email_verified_at ? 'Yes' : 'No',
            'created_at' => optional($user->created_at)->format('Y-m-d'),
        ];

        $filename = 'users-' . now()->format('Ymd-His');

        return $exporter->download((string) $request->query('format', 'csv'), $filename, $query, $columns, $map, 'User Report', 'User export');
    }
}
