<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\LoginRequest;
use App\Models\User;
use App\Support\ActivityLogger;
use App\Support\SecurityAuditLogger;
use App\Support\SecuritySettings;
use App\Support\SocialLoginSettings;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;

class AuthenticatedSessionController extends Controller
{
    /**
     * Display the login view.
     */
    public function create(): View
    {
        return view('auth.login', [
            'captchaEnabled' => SecuritySettings::getValue('captcha_enabled', false),
            'captchaSiteKey' => SecuritySettings::getValue('captcha_site_key', ''),
            'socialProviders' => SocialLoginSettings::enabledProviders(),
        ]);
    }

    /**
     * Handle an incoming authentication request.
     */
    public function store(LoginRequest $request): RedirectResponse
    {
        $request->authenticate();

        $request->session()->regenerate();

        $user = $request->user();

        if ($user instanceof User && $this->shouldRequireTwoFactor($user)) {
            $requiresSetup = ! $user->two_factor_enabled;

            $request->session()->put('two_factor_user_id', $user->id);
            $request->session()->put('two_factor_remember', $request->boolean('remember'));
            $request->session()->put('two_factor_setup', $requiresSetup);

            SecurityAuditLogger::log('two_factor_required', 'info', $user, $request, [
                'setup_required' => $requiresSetup,
            ]);

            Auth::guard('web')->logout();

            return redirect()->route($requiresSetup ? 'two-factor.setup' : 'two-factor.challenge');
        }

        if ($user instanceof User) {
            SecurityAuditLogger::log('login_success', 'success', $user, $request);
            ActivityLogger::logFromRequest($request, 'login', 'User', $user->id, 'Logged in');
        }

        return redirect()->intended(route('dashboard', absolute: false));
    }

    /**
     * Destroy an authenticated session.
     */
    public function destroy(Request $request): RedirectResponse
    {
        $user = $request->user();

        Auth::guard('web')->logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        if ($user) {
            ActivityLogger::logFromRequest($request, 'logout', 'User', $user->id, 'Logged out');
        }

        return redirect('/');
    }

    private function shouldRequireTwoFactor(User $user): bool
    {
        $settings = SecuritySettings::get();
        $adminOnly = (bool) ($settings['two_factor_admin_only'] ?? false);

        if (! $adminOnly) {
            return $user->two_factor_enabled;
        }

        $adminRoles = config('security.admin_roles', []);

        return $user->hasAnyRole($adminRoles) || $user->two_factor_enabled;
    }
}
