<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\SocialAccount;
use App\Models\User;
use App\Support\ActivityLogger;
use App\Support\SecurityAuditLogger;
use App\Support\SecuritySettings;
use App\Support\SocialLoginSettings;
use App\Support\UserDefaults;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Laravel\Socialite\Facades\Socialite;
use Throwable;

class SocialLoginController extends Controller
{
    public function redirect(Request $request, string $provider): RedirectResponse
    {
        $config = SocialLoginSettings::providerConfig($provider);

        if (! $this->isProviderReady($config)) {
            abort(404);
        }

        $this->configureProvider($provider, $config);

        return Socialite::driver($config['driver'])->redirect();
    }

    public function callback(Request $request, string $provider): RedirectResponse
    {
        $config = SocialLoginSettings::providerConfig($provider);

        if (! $this->isProviderReady($config)) {
            return redirect()
                ->route('login')
                ->with('error', 'Social login is not available for this provider.');
        }

        $this->configureProvider($provider, $config);

        try {
            $socialUser = Socialite::driver($config['driver'])->user();
        } catch (Throwable $exception) {
            return redirect()
                ->route('login')
                ->with('error', 'Unable to authenticate with the social provider.');
        }

        $account = SocialAccount::query()
            ->where('provider', $provider)
            ->where('provider_id', $socialUser->getId())
            ->first();

        if ($account) {
            $user = $account->user;
        } else {
            $email = $socialUser->getEmail();

            if (! $email) {
                return redirect()
                    ->route('login')
                    ->with('error', 'Your provider did not return an email address.');
            }

            $user = User::query()->where('email', $email)->first();

            if (! $user) {
                // New user - create and link
                $user = User::create([
                    'name' => $socialUser->getName() ?: $socialUser->getNickname() ?: 'Social User',
                    'email' => $email,
                    'email_verified_at' => now(),
                    'password' => Str::random(32),
                    'is_active' => true,
                ]);

                UserDefaults::assignDefaults($user);

                SocialAccount::create([
                    'provider' => $provider,
                    'provider_id' => $socialUser->getId(),
                    'user_id' => $user->id,
                    'provider_email' => $socialUser->getEmail(),
                    'avatar' => $socialUser->getAvatar(),
                ]);
            } else {
                // Existing user found by email - only auto-link if both sides have verified email
                $providerEmailVerified = ! empty($socialUser->user['email_verified'] ?? $socialUser->user['verified_email'] ?? false);

                if (! $user->email_verified_at || ! $providerEmailVerified) {
                    SecurityAuditLogger::log('social_link_denied', 'warning', $user, $request, [
                        'provider' => $provider,
                        'reason' => 'Email not verified on both sides',
                    ]);

                    return redirect()
                        ->route('login')
                        ->with('error', 'An account with this email already exists. Please log in with your password first, then link your social account.');
                }

                SocialAccount::create([
                    'provider' => $provider,
                    'provider_id' => $socialUser->getId(),
                    'user_id' => $user->id,
                    'provider_email' => $socialUser->getEmail(),
                    'avatar' => $socialUser->getAvatar(),
                ]);
            }
        }

        if (! $user) {
            return redirect()
                ->route('login')
                ->with('error', 'Unable to sign you in with the social provider.');
        }

        Auth::login($user, true);

        if ($this->shouldRequireTwoFactor($user)) {
            $requiresSetup = ! $user->two_factor_enabled;
            $request->session()->put('two_factor_user_id', $user->id);
            $request->session()->put('two_factor_remember', true);
            $request->session()->put('two_factor_setup', $requiresSetup);

            SecurityAuditLogger::log('two_factor_required', 'info', $user, $request, [
                'setup_required' => $requiresSetup,
            ]);

            Auth::guard('web')->logout();

            return redirect()->route($requiresSetup ? 'two-factor.setup' : 'two-factor.challenge');
        }

        SecurityAuditLogger::log('login_success', 'success', $user, $request, [
            'provider' => $provider,
            'social_login' => true,
        ]);
        ActivityLogger::logFromRequest($request, 'login', 'User', $user->id, 'Logged in with ' . $provider);

        return redirect()->intended(route('dashboard', absolute: false));
    }

    private function configureProvider(string $provider, array $config): void
    {
        $redirectUrl = $config['redirect_url'] ?: route('social.callback', $provider);

        config([
            'services.' . $config['driver'] => [
                'client_id' => $config['client_id'],
                'client_secret' => $config['client_secret'],
                'redirect' => $redirectUrl,
            ],
        ]);
    }

    private function isProviderReady(?array $config): bool
    {
        if (! $config) {
            return false;
        }

        if (! $config['enabled']) {
            return false;
        }

        return $config['client_id'] !== '' && $config['client_secret'] !== '';
    }

    private function shouldRequireTwoFactor(User $user): bool
    {
        $settings = SecuritySettings::get();
        $adminOnly = (bool) ($settings['two_factor_admin_only'] ?? false);

        if (! $adminOnly) {
            return $user->two_factor_enabled;
        }

        $adminRoles = config('security.admin_roles', []);

        return $user->hasAnyRole($adminRoles) || $user->two_factor_enabled;
    }
}
