<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Support\ActivityLogger;
use App\Support\SecurityAuditLogger;
use App\Support\SecuritySettings;
use App\Support\TwoFactorService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;

class TwoFactorChallengeController extends Controller
{
    public function create(Request $request): View|RedirectResponse
    {
        $user = $this->resolveUser($request);

        if (! $user || (bool) $request->session()->get('two_factor_setup')) {
            return redirect()->route('login')->with('error', 'Two-factor verification is not available.');
        }

        return view('auth.two-factor-challenge', [
            'email' => $user->email,
        ]);
    }

    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'code' => ['required', 'string'],
        ]);

        $user = $this->resolveUser($request);

        if (! $user || ! $user->two_factor_secret) {
            return redirect()->route('login')->with('error', 'Two-factor verification expired. Please sign in again.');
        }

        if (! TwoFactorService::verifyCode($user->two_factor_secret, $request->input('code'))) {
            SecurityAuditLogger::log('two_factor_failed', 'failed', $user, $request);

            return back()->withErrors([
                'code' => 'The authentication code is invalid.',
            ]);
        }

        $this->finalizeLogin($request, $user);

        SecurityAuditLogger::log('two_factor_verified', 'success', $user, $request);
        SecurityAuditLogger::log('login_success', 'success', $user, $request);
        ActivityLogger::logFromRequest($request, 'login', 'User', $user->id, 'Logged in with 2FA');

        return redirect()->intended(route('dashboard', absolute: false));
    }

    public function setup(Request $request): View|RedirectResponse
    {
        $user = $this->resolveUser($request);

        if (! $user) {
            return redirect()->route('login')->with('error', 'Two-factor setup expired. Please sign in again.');
        }

        $secret = $request->session()->get('two_factor_setup_secret');
        if (! $secret) {
            $secret = TwoFactorService::generateSecret();
            $request->session()->put('two_factor_setup_secret', $secret);
        }

        $issuer = SecuritySettings::getValue('two_factor_issuer', config('app.name', 'SnapNest'));
        $otpAuthUrl = TwoFactorService::buildOtpAuthUrl($user, $secret, $issuer);

        return view('auth.two-factor-setup', [
            'secret' => $secret,
            'otpAuthUrl' => $otpAuthUrl,
            'issuer' => $issuer,
            'email' => $user->email,
        ]);
    }

    public function setupStore(Request $request): RedirectResponse
    {
        $request->validate([
            'code' => ['required', 'string'],
        ]);

        $user = $this->resolveUser($request);

        if (! $user) {
            return redirect()->route('login')->with('error', 'Two-factor setup expired. Please sign in again.');
        }

        $secret = $request->session()->get('two_factor_setup_secret');

        if (! $secret) {
            return back()->with('error', 'Two-factor setup expired. Please restart setup.');
        }

        if (! TwoFactorService::verifyCode($secret, $request->input('code'))) {
            SecurityAuditLogger::log('two_factor_failed', 'failed', $user, $request);

            return back()->withErrors([
                'code' => 'The authentication code is invalid.',
            ]);
        }

        $user->forceFill([
            'two_factor_enabled' => true,
            'two_factor_secret' => $secret,
            'two_factor_confirmed_at' => now(),
        ])->save();

        $request->session()->forget([
            'two_factor_setup_secret',
            'two_factor_setup',
        ]);

        $this->finalizeLogin($request, $user);

        SecurityAuditLogger::log('two_factor_enabled', 'success', $user, $request);
        SecurityAuditLogger::log('login_success', 'success', $user, $request);
        ActivityLogger::logFromRequest($request, 'login', 'User', $user->id, 'Logged in with 2FA setup');

        return redirect()->intended(route('dashboard', absolute: false))
            ->with('status', 'Two-factor authentication enabled.');
    }

    private function resolveUser(Request $request): ?User
    {
        $userId = $request->session()->get('two_factor_user_id');

        if (! $userId) {
            return null;
        }

        return User::query()->find($userId);
    }

    private function finalizeLogin(Request $request, User $user): void
    {
        $remember = (bool) $request->session()->pull('two_factor_remember', false);

        Auth::login($user, $remember);

        $request->session()->forget([
            'two_factor_user_id',
            'two_factor_setup',
        ]);

        $request->session()->regenerate();
    }
}
