<?php

namespace App\Http\Controllers;

use App\Models\Coupon;
use App\Models\Currency;
use App\Models\Order;
use App\Models\PaymentGateway;
use App\Models\Plan;
use App\Mail\OrderCompletedMail;
use App\Services\Payment\PaymentGatewayManager;
use App\Support\FormatSettings;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;

class CheckoutController extends Controller
{
    public function __construct(
        private readonly PaymentGatewayManager $gatewayManager,
    ) {}

    // -----------------------------------------------------------------
    //  Show the checkout page for a specific plan
    // -----------------------------------------------------------------

    public function show(Request $request, Plan $plan)
    {
        if (! $plan->is_active) {
            abort(404);
        }

        // Free plans go straight to registration
        if ((float) $plan->price === 0.0) {
            return redirect()->route('register');
        }

        $gateways = PaymentGateway::active()->orderBy('sort_order')->get();

        $effectivePrice = $plan->offer_price !== null ? $plan->offer_price : $plan->price;

        // Resolve default currency from system settings
        $currencyConfig = FormatSettings::currencyConfig();
        $defaultCurrency = $currencyConfig['code'] ?: 'USD';
        $defaultSymbol = $currencyConfig['symbol'] ?: '$';

        // Get all active currencies for dropdown
        $currencies = Currency::where('is_active', true)->orderBy('name')->get();

        // Build gateway → currencies map for JS filtering
        $gatewayCurrencies = [];
        foreach ($gateways as $gw) {
            $gatewayCurrencies[$gw->slug] = $gw->supported_currencies ?? [];
        }

        // Build currency data map for JS (symbol, exchange_rate, precision)
        $currencyData = [];
        $defaultRate = 1;
        foreach ($currencies as $cur) {
            $currencyData[$cur->code] = [
                'symbol'    => $cur->symbol,
                'rate'      => (float) $cur->exchange_rate,
                'precision' => $cur->precision,
            ];
            if ($cur->code === $defaultCurrency) {
                $defaultRate = (float) $cur->exchange_rate;
            }
        }

        return view('checkout.index', [
            'plan'              => $plan,
            'gateways'          => $gateways,
            'effectivePrice'    => $effectivePrice,
            'currency'          => $defaultCurrency,
            'currencySymbol'    => $defaultSymbol,
            'currencies'        => $currencies,
            'gatewayCurrencies' => $gatewayCurrencies,
            'currencyData'      => $currencyData,
            'defaultRate'       => $defaultRate,
        ]);
    }

    // -----------------------------------------------------------------
    //  Process the checkout -- create order and initiate payment
    // -----------------------------------------------------------------

    public function process(Request $request, Plan $plan)
    {
        if (! $plan->is_active) {
            abort(404);
        }

        $request->validate([
            'gateway'           => ['required', 'string'],
            'coupon_code'       => ['nullable', 'string', 'max:100'],
            'selected_currency' => ['nullable', 'string', 'max:10'],
        ]);

        $user          = $request->user();
        $gatewaySlug   = $request->input('gateway');
        $couponCode    = $request->input('coupon_code');
        $effectivePrice = $plan->offer_price !== null ? (float) $plan->offer_price : (float) $plan->price;

        // Resolve selected currency and exchange rate
        $currencyConfig  = FormatSettings::currencyConfig();
        $defaultCode     = $currencyConfig['code'] ?: 'USD';
        $selectedCode    = strtoupper(trim($request->input('selected_currency', $defaultCode)));

        // Look up rates from DB (never trust client-side rates)
        $defaultCurrency  = Currency::where('code', $defaultCode)->where('is_active', true)->first();
        $selectedCurrency = Currency::where('code', $selectedCode)->where('is_active', true)->first();

        // Fall back to default if selected currency is invalid
        if (! $selectedCurrency) {
            $selectedCurrency = $defaultCurrency;
            $selectedCode     = $defaultCode;
        }

        $defaultRate  = $defaultCurrency ? (float) $defaultCurrency->exchange_rate : 1;
        $selectedRate = (float) $selectedCurrency->exchange_rate;
        $selectedSymbol = $selectedCurrency->symbol ?: '$';

        // Validate the gateway exists and is active
        $gateway = PaymentGateway::where('slug', $gatewaySlug)->where('is_active', true)->first();
        if (! $gateway) {
            return back()->withErrors(['gateway' => 'Selected payment gateway is not available.'])->withInput();
        }

        // Apply coupon if provided
        $coupon       = null;
        $discount     = 0;
        $finalAmount  = $effectivePrice;

        if ($couponCode) {
            $couponResult = $this->validateCoupon($couponCode, $plan, $effectivePrice, $user);

            if ($couponResult['valid']) {
                $coupon      = $couponResult['coupon'];
                $discount    = $couponResult['discount'];
                $finalAmount = $couponResult['final_amount'];
            } else {
                return back()->withErrors(['coupon_code' => $couponResult['message']])->withInput();
            }
        }

        // Prevent zero-amount payments from going to a gateway
        if ($finalAmount <= 0) {
            return $this->handleFreeOrder($user, $plan, $coupon, $effectivePrice, $discount);
        }

        // Convert amount to the selected currency
        $convertedFinalAmount = $defaultRate > 0
            ? round(($finalAmount / $defaultRate) * $selectedRate, 2)
            : $finalAmount;

        // Create the order
        $order = DB::transaction(function () use ($user, $plan, $coupon, $convertedFinalAmount, $gateway, $effectivePrice, $discount, $selectedCode, $defaultCode, $defaultRate, $selectedRate, $finalAmount) {
            $order = Order::create([
                'user_id'            => $user->id,
                'plan_id'            => $plan->id,
                'coupon_id'          => $coupon?->id,
                'order_number'       => 'ORD-' . strtoupper(Str::random(12)),
                'amount'             => $convertedFinalAmount,
                'currency'           => $selectedCode,
                'status'             => 'pending',
                'payment_method'     => $gateway->name,
                'payment_gateway_id' => $gateway->id,
                'metadata'           => [
                    'plan_name'         => $plan->name,
                    'original_price'    => $effectivePrice,
                    'discount'          => $discount,
                    'coupon_code'       => $coupon?->code,
                    'base_currency'     => $defaultCode,
                    'base_amount'       => $finalAmount,
                    'exchange_rate'     => $selectedRate,
                    'base_rate'         => $defaultRate,
                ],
            ]);

            // Increment coupon usage
            if ($coupon) {
                $coupon->increment('current_uses');
            }

            return $order;
        });

        // Initiate payment via the gateway driver
        try {
            $driver      = $this->gatewayManager->driverFromModel($gateway);
            $callbackUrl = route('payment.callback', [
                'gateway'   => $gateway->slug,
                'order_ref' => $order->order_number,
            ]);
            $initiateResult = $driver->initiate($order, $callbackUrl);

            // Store in session as primary lookup; order_ref in URL is the fallback
            session(['pending_order_id' => $order->id]);

            if (isset($initiateResult['redirect_url'])) {
                return redirect()->away($initiateResult['redirect_url']);
            }

            if (isset($initiateResult['html'])) {
                return view('checkout.gateway-form', [
                    'html'  => $initiateResult['html'],
                    'order' => $order,
                    'plan'  => $plan,
                ]);
            }

            return back()->withErrors(['gateway' => 'Payment gateway did not return a valid response.'])->withInput();
        } catch (\Exception $e) {
            Log::error('Payment initiation failed', [
                'order_id' => $order->id,
                'gateway'  => $gateway->slug,
                'error'    => $e->getMessage(),
            ]);

            $order->update(['status' => 'failed', 'metadata' => array_merge($order->metadata ?? [], ['error' => $e->getMessage()])]);

            return back()->withErrors(['gateway' => 'Payment initiation failed. Please try again or choose a different payment method.'])->withInput();
        }
    }

    // -----------------------------------------------------------------
    //  Handle return from payment gateway (GET/POST callback)
    // -----------------------------------------------------------------

    public function callback(Request $request, string $gateway)
    {
        $orderId  = session('pending_order_id');
        $orderRef = $request->input('order_ref');

        if ($orderId) {
            $order = Order::with('plan')->find($orderId);
        } elseif ($orderRef) {
            // Session expired (user opened new tab / browser restarted) — fall back to URL ref
            $order = Order::with('plan')
                ->where('order_number', $orderRef)
                ->where('user_id', auth()->id())
                ->first();
        } else {
            return redirect()->route('checkout.cancel')->with('error', 'Payment session expired. Please try again.');
        }

        if (! $order) {
            return redirect()->route('checkout.cancel')->with('error', 'Order not found.');
        }

        // Check for explicit cancellation
        if ($request->has('cancelled') || $request->has('cancel')) {
            $order->update(['status' => 'failed']);
            session()->forget('pending_order_id');
            return redirect()->route('checkout.cancel');
        }

        // Already completed (e.g. webhook arrived first)
        if ($order->status === 'completed') {
            session()->forget('pending_order_id');
            return redirect()->route('checkout.success', $order);
        }

        try {
            $gatewayModel = $order->paymentGateway ?? PaymentGateway::where('slug', $gateway)->firstOrFail();
            $driver       = $this->gatewayManager->driverFromModel($gatewayModel);
            $result       = $driver->handleCallback($request->all());

            if ($result->success) {
                $this->completeOrder($order, $result->transactionId, $result->metadata);
                session()->forget('pending_order_id');
                return redirect()->route('checkout.success', $order);
            }

            // Pending verification (bank transfer, offline, etc.)
            if ($result->isPending()) {
                $order->update([
                    'status'                 => 'pending_verification',
                    'gateway_transaction_id' => $result->transactionId,
                    'metadata'               => array_merge($order->metadata ?? [], $result->metadata),
                ]);

                session()->forget('pending_order_id');
                return redirect()->route('checkout.success', $order)->with('payment_pending', $result->message);
            }

            // Payment failed
            $order->update([
                'status'   => 'failed',
                'metadata' => array_merge($order->metadata ?? [], ['callback_error' => $result->message]),
            ]);

            session()->forget('pending_order_id');
            return redirect()->route('checkout.cancel')->with('error', $result->message);
        } catch (\Exception $e) {
            Log::error('Payment callback processing failed', [
                'order_id' => $order->id,
                'gateway'  => $gateway,
                'error'    => $e->getMessage(),
            ]);

            session()->forget('pending_order_id');
            return redirect()->route('checkout.cancel')->with('error', 'Payment verification failed. If you were charged, please contact support.');
        }
    }

    // -----------------------------------------------------------------
    //  Handle async webhook from payment gateway
    // -----------------------------------------------------------------

    public function webhook(Request $request, string $gateway)
    {
        Log::info('Payment webhook received', ['gateway' => $gateway]);

        try {
            $gatewayModel = PaymentGateway::where('slug', $gateway)->where('is_active', true)->firstOrFail();
            $driver       = $this->gatewayManager->driverFromModel($gatewayModel);

            // Verify webhook signature before processing
            $rawBody = $request->getContent();
            $headers = collect($request->headers->all())->map(fn ($v) => $v[0] ?? '')->toArray();

            if (! $driver->verifyWebhookSignature($rawBody, $headers)) {
                Log::warning('Webhook signature verification failed', ['gateway' => $gateway]);
                return response()->json(['status' => 'invalid_signature'], 403);
            }

            $result = $driver->handleWebhook($request->all());

            if (! $result->success && ! $result->isPending()) {
                Log::warning('Webhook processing returned failure', ['gateway' => $gateway, 'message' => $result->message]);
                return response()->json(['status' => 'ignored', 'message' => $result->message], 200);
            }

            // Find the order by transaction ID or metadata
            $order = $this->resolveOrderFromWebhook($result, $gatewayModel);

            if (! $order) {
                Log::warning('Webhook: order not found', ['gateway' => $gateway, 'transaction_id' => $result->transactionId]);
                return response()->json(['status' => 'order_not_found'], 200);
            }

            // Idempotency: skip if already completed
            if ($order->status === 'completed') {
                return response()->json(['status' => 'already_completed'], 200);
            }

            // Handle pending results (e.g. bank transfer awaiting verification)
            if ($result->isPending()) {
                if ($order->status !== 'pending_verification') {
                    $order->update([
                        'status'                 => 'pending_verification',
                        'gateway_transaction_id' => $result->transactionId,
                        'metadata'               => array_merge($order->metadata ?? [], $result->metadata),
                    ]);
                }
                return response()->json(['status' => 'pending'], 200);
            }

            $this->completeOrder($order, $result->transactionId, $result->metadata);

            return response()->json(['status' => 'ok'], 200);
        } catch (\Exception $e) {
            Log::error('Payment webhook processing failed', ['gateway' => $gateway, 'error' => $e->getMessage()]);
            return response()->json(['status' => 'error'], 500);
        }
    }

    // -----------------------------------------------------------------
    //  Success page
    // -----------------------------------------------------------------

    public function success(Order $order)
    {
        // Ensure the user can only see their own order
        if ($order->user_id !== auth()->id()) {
            abort(403);
        }

        $order->load('plan');

        // Resolve currency symbol from the order's currency
        $currency = Currency::where('code', $order->currency)->first();
        $currencySymbol = $currency?->symbol ?: (FormatSettings::currencyConfig()['symbol'] ?: '$');

        return view('checkout.success', [
            'order'          => $order,
            'currencySymbol' => $currencySymbol,
        ]);
    }

    // -----------------------------------------------------------------
    //  Cancel page
    // -----------------------------------------------------------------

    public function cancel()
    {
        return view('checkout.cancel');
    }

    // -----------------------------------------------------------------
    //  AJAX coupon validation
    // -----------------------------------------------------------------

    public function applyCoupon(Request $request): JsonResponse
    {
        $request->validate([
            'coupon_code' => ['required', 'string', 'max:100'],
            'plan_id'     => ['required', 'integer', 'exists:plans,id'],
        ]);

        $plan = Plan::findOrFail($request->input('plan_id'));
        $effectivePrice = $plan->offer_price !== null ? (float) $plan->offer_price : (float) $plan->price;
        $user = $request->user();

        $result = $this->validateCoupon($request->input('coupon_code'), $plan, $effectivePrice, $user);

        if ($result['valid']) {
            return response()->json([
                'valid'        => true,
                'discount'     => $result['discount'],
                'final_amount' => $result['final_amount'],
                'message'      => 'Coupon applied successfully.',
            ]);
        }

        return response()->json([
            'valid'   => false,
            'message' => $result['message'],
        ], 422);
    }

    // =================================================================
    //  Private helpers
    // =================================================================

    private function validateCoupon(string $code, Plan $plan, float $price, $user): array
    {
        $coupon = Coupon::where('code', $code)->where('is_active', true)->first();

        if (! $coupon) {
            return ['valid' => false, 'message' => 'Invalid coupon code.'];
        }

        // Date validity
        $now = now();
        if ($coupon->starts_at && $now->lt($coupon->starts_at)) {
            return ['valid' => false, 'message' => 'This coupon is not yet active.'];
        }
        if ($coupon->ends_at && $now->gt($coupon->ends_at)) {
            return ['valid' => false, 'message' => 'This coupon has expired.'];
        }

        // Usage limits
        if ($coupon->max_uses !== null && $coupon->current_uses >= $coupon->max_uses) {
            return ['valid' => false, 'message' => 'This coupon has reached its usage limit.'];
        }

        // New users only
        if ($coupon->new_users_only && $user && Order::where('user_id', $user->id)->completed()->exists()) {
            return ['valid' => false, 'message' => 'This coupon is only available for new users.'];
        }

        // Minimum amount
        if ($coupon->min_amount && $price < (float) $coupon->min_amount) {
            $sym = FormatSettings::currencyConfig()['symbol'] ?: '$';
            return ['valid' => false, 'message' => 'Minimum order amount of ' . $sym . number_format($coupon->min_amount, 2) . ' required.'];
        }

        // Plan applicability -- if the coupon has specific plans, check the pivot
        if ($coupon->plans()->count() > 0 && ! $coupon->plans()->where('plans.id', $plan->id)->exists()) {
            return ['valid' => false, 'message' => 'This coupon is not valid for the selected plan.'];
        }

        // Calculate discount
        $discount = 0;
        if ($coupon->discount_type === 'percentage') {
            $discount = round($price * ((float) $coupon->discount_value / 100), 2);
        } else {
            // fixed
            $discount = min((float) $coupon->discount_value, $price);
        }

        $finalAmount = max(0, round($price - $discount, 2));

        return [
            'valid'        => true,
            'coupon'       => $coupon,
            'discount'     => $discount,
            'final_amount' => $finalAmount,
        ];
    }

    private function completeOrder(Order $order, ?string $transactionId, array $metadata = []): void
    {
        DB::transaction(function () use ($order, $transactionId, $metadata) {
            $order->update([
                'status'                 => 'completed',
                'gateway_transaction_id' => $transactionId,
                'metadata'               => array_merge($order->metadata ?? [], $metadata),
            ]);

            // Assign the purchased plan to the user
            if ($order->plan_id) {
                $order->user()->update(['plan_id' => $order->plan_id]);
            }
        });

        // Send payment confirmation email (outside transaction — failure won't roll back the order)
        try {
            $order->loadMissing(['user', 'plan']);
            if ($order->user?->email) {
                Mail::to($order->user->email)->send(new OrderCompletedMail($order));
            }
        } catch (\Throwable $e) {
            Log::warning('Failed to send order confirmation email', [
                'order_id' => $order->id,
                'error'    => $e->getMessage(),
            ]);
        }
    }

    private function handleFreeOrder($user, Plan $plan, ?Coupon $coupon, float $originalPrice, float $discount)
    {
        $currencyConfig = FormatSettings::currencyConfig();
        $defaultCode    = $currencyConfig['code'] ?: 'USD';

        $order = DB::transaction(function () use ($user, $plan, $coupon, $originalPrice, $discount, $defaultCode) {
            $order = Order::create([
                'user_id'        => $user->id,
                'plan_id'        => $plan->id,
                'coupon_id'      => $coupon?->id,
                'order_number'   => 'ORD-' . strtoupper(Str::random(12)),
                'amount'         => 0,
                'currency'       => $defaultCode,
                'status'         => 'completed',
                'payment_method' => 'coupon',
                'metadata'       => [
                    'plan_name'      => $plan->name,
                    'original_price' => $originalPrice,
                    'discount'       => $discount,
                    'coupon_code'    => $coupon?->code,
                    'free_via_coupon' => true,
                ],
            ]);

            if ($coupon) {
                $coupon->increment('current_uses');
            }

            // Assign plan immediately
            $user->update(['plan_id' => $plan->id]);

            return $order;
        });

        return redirect()->route('checkout.success', $order);
    }

    private function resolveOrderFromWebhook($result, PaymentGateway $gateway): ?Order
    {
        // Try to find by transaction ID first
        if ($result->transactionId) {
            $order = Order::where('gateway_transaction_id', $result->transactionId)->first();
            if ($order) {
                return $order;
            }
        }

        // Try metadata for order number
        $orderNumber = $result->metadata['order_number'] ?? null;
        if ($orderNumber) {
            return Order::where('order_number', $orderNumber)->first();
        }

        // Try Stripe-specific: metadata contains order_id
        $orderId = $result->metadata['stripe_session_id'] ?? $result->metadata['order_id'] ?? null;
        if ($orderId && is_numeric($orderId)) {
            return Order::find($orderId);
        }

        return null;
    }
}
