<?php

namespace App\Http\Controllers;

use App\Jobs\ProcessEventMedia;
use App\Models\Event;
use App\Models\EventMedia;
use App\Services\FaceRecognitionService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;

class EventShareController extends Controller
{
    public function show(Request $request, Event $event)
    {
        if (! $this->tokenMatches($request, $event)) {
            abort(404);
        }

        if (! $event->is_active) {
            return view('events.share', [
                'event' => $event,
                'media' => collect(),
                'requiresPin' => false,
                'pinRequired' => false,
                'expired' => $this->isExpired($event),
                'inactive' => true,
                'mediaStats' => $this->emptyMediaStats(),
                'faceSearchRequired' => false,
                'faceSearchReady' => true,
                'faceSearchImageUrl' => null,
            ]);
        }

        $isPreview = $request->query('design_preview') === 'true'
            && $request->user()
            && ($request->user()->is_admin || $request->user()->id === $event->created_by);
        $requiresPin = $isPreview ? false : $this->requiresPin($event);
        $hasAccess = $requiresPin ? $request->session()->get($this->pinSessionKey($event), false) : true;

        if ($isPreview) {
            $hasAccess = true;
        }

        $isAdminAccess = $request->session()->get($this->pinTypeSessionKey($event)) === 'admin';
        $faceAvailable = app(FaceRecognitionService::class)->isAvailable();
        $faceSearchRequired = ($isPreview || $isAdminAccess || $event->allow_public_downloads || ! $faceAvailable) ? false : $requiresPin;
        $faceSearchReady = $faceSearchRequired
            ? (bool) $request->session()->get($this->faceSessionKey($event), false)
            : true;
        // R9: Face images now stored on private disk, served via signed URL
        $faceSearchImage = $request->session()->get($this->faceSessionImageKey($event));
        $faceSearchImageUrl = $faceSearchImage
            ? route('events.share.face-image', ['event' => $event->slug, 'token' => $event->share_token])
            : null;

        // FIX #8: Validate face search session hasn't expired (30 minutes)
        $faceSearchData = $request->session()->get($this->faceSessionMatchesKey($event));
        $faceSearchMatches = [];

        if ($faceSearchReady && ! $isPreview && $faceSearchData && is_array($faceSearchData)) {
            $matchedAt = $faceSearchData['matched_at'] ?? 0;
            $expiryMinutes = 30;

            if (now()->timestamp - $matchedAt < ($expiryMinutes * 60)) {
                $faceSearchMatches = (array) ($faceSearchData['matched_ids'] ?? []);
                $faceSearchMatches = array_values(array_filter(array_map('intval', $faceSearchMatches)));
            } else {
                // Session expired - clear it
                $request->session()->forget($this->faceSessionMatchesKey($event));
                $request->session()->forget($this->faceSessionKey($event));
            }
        }

        $media = collect();
        $mediaPaginator = null;
        $mediaStats = $this->emptyMediaStats();
        $perPage = 40;

        // Track gallery view (skip on AJAX load-more)
        if ($hasAccess && $faceSearchReady && ! $request->ajax()) {
            app(\App\Services\AnalyticsService::class)->recordView($event);
        }

        if ($hasAccess && $faceSearchReady && Schema::hasTable('event_media')) {
            $mediaQuery = $event->media()->whereIn('status', ['ready', 'processing', 'pending']);

            // Get total stats before pagination
            $statsQuery = (clone $mediaQuery);

            if ($faceSearchRequired) {
                if ($faceSearchMatches) {
                    $safeIds = array_values(array_filter(array_map('intval', $faceSearchMatches)));
                    $mediaQuery->whereIn('id', $safeIds);
                    $statsQuery->whereIn('id', $safeIds);
                } else {
                    $mediaQuery->whereRaw('1=0');
                    $statsQuery->whereRaw('1=0');
                }
            }

            // Stats from full (unpaginated) set
            $allForStats = $statsQuery->get(['id', 'status']);
            $mediaStats = [
                'total' => $allForStats->count(),
                'ready' => $allForStats->where('status', 'ready')->count(),
                'processing' => $allForStats->where('status', 'processing')->count(),
                'pending' => $allForStats->where('status', 'pending')->count(),
            ];

            if ($faceSearchRequired && $faceSearchMatches) {
                // Face search: fetch all matches, sort by match order, then manual paginate
                $allMedia = $mediaQuery->get();
                $order = array_flip(array_map('intval', $faceSearchMatches));
                $allMedia = $allMedia->sort(function ($a, $b) use ($order) {
                    return ($order[$a->id] ?? PHP_INT_MAX) <=> ($order[$b->id] ?? PHP_INT_MAX);
                })->values();

                $page = max(1, (int) $request->query('page', 1));
                $slice = $allMedia->slice(($page - 1) * $perPage, $perPage)->values();
                $mediaPaginator = new \Illuminate\Pagination\LengthAwarePaginator(
                    $slice, $allMedia->count(), $perPage, $page,
                    ['path' => $request->url(), 'query' => $request->query()]
                );
                $media = $slice;
            } else {
                // Normal: use DB pagination
                $mediaQuery->orderByDesc('created_at');
                $mediaPaginator = $mediaQuery->paginate($perPage)->withQueryString();
                $media = $mediaPaginator->getCollection();
            }
        }

        // AJAX load-more: return JSON
        if ($request->ajax() || $request->wantsJson()) {
            $token = (string) $request->query('token', '');
            $items = $media->map(fn ($item) => [
                'id' => $item->id,
                'preview_url' => $item->signedThumbnailUrl(240) ?: $item->signedPreviewUrl(240),
                'full_url' => $item->signedPreviewUrl(240),
                'file_type' => $item->file_type,
                'is_guest' => (bool) $item->is_guest_upload,
                'is_featured' => (bool) $item->is_featured,
                'download_url' => route('events.share.download', [
                    'event' => $event->slug, 'media' => $item->id, 'token' => $token,
                ]),
            ]);

            return response()->json([
                'items' => $items,
                'has_more' => $mediaPaginator?->hasMorePages() ?? false,
                'total' => $mediaPaginator?->total() ?? 0,
                'current_page' => $mediaPaginator?->currentPage() ?? 1,
                'last_page' => $mediaPaginator?->lastPage() ?? 1,
            ]);
        }

        return view('events.share', [
            'event' => $event,
            'media' => $media,
            'mediaPaginator' => $mediaPaginator,
            'requiresPin' => $requiresPin,
            'pinRequired' => $requiresPin && ! $hasAccess,
            'expired' => $this->isExpired($event),
            'inactive' => false,
            'mediaStats' => $mediaStats,
            'faceSearchRequired' => $faceSearchRequired,
            'faceSearchReady' => $faceSearchReady,
            'faceSearchImageUrl' => $faceSearchImageUrl,
            'faceSearchMatchesCount' => count($faceSearchMatches),
            'showBranding' => $this->shouldShowBranding($event),
        ]);
    }

    public function verifyPin(Request $request, Event $event)
    {
        if (! $this->tokenMatches($request, $event)) {
            abort(404);
        }

        $request->validate([
            'pin' => ['required', 'string', 'regex:/^[0-9]{4,8}$/'],
        ]);

        $pinType = $this->pinMatches($event, $request->input('pin'));

        if (! $pinType) {
            return back()->withErrors([
                'pin' => 'Invalid pin. Please try again.',
            ]);
        }

        $request->session()->put($this->pinSessionKey($event), true);
        $request->session()->put($this->pinTypeSessionKey($event), $pinType);

        return redirect()->route('events.share', [
            'event' => $event->slug,
            'token' => $event->share_token,
        ]);
    }

    public function faceSearch(Request $request, Event $event, FaceRecognitionService $faceRecognition)
    {
        if (! $this->tokenMatches($request, $event)) {
            abort(404);
        }

        // FIX #7: Rate limiting to prevent CPU abuse
        $rateLimitKey = 'face-search:' . $event->id . ':' . $request->ip();

        if (RateLimiter::tooManyAttempts($rateLimitKey, 5)) {
            $seconds = RateLimiter::availableIn($rateLimitKey);

            return back()->with('error', "Too many face search attempts. Please wait {$seconds} seconds before trying again.");
        }

        RateLimiter::hit($rateLimitKey, 60); // 5 per minute

        if (! $event->is_active || $this->isExpired($event)) {
            return redirect()->route('events.share', [
                'event' => $event->slug,
                'token' => $event->share_token,
            ]);
        }

        if ($this->requiresPin($event) && ! $request->session()->get($this->pinSessionKey($event), false)) {
            return redirect()->route('events.share', [
                'event' => $event->slug,
                'token' => $event->share_token,
            ]);
        }

        $validated = $request->validate([
            'face_image' => ['required', 'image', 'mimes:jpg,jpeg,png,webp', 'max:5120'],
        ]);

        $previous = $request->session()->get($this->faceSessionImageKey($event));
        if ($previous) {
            Storage::disk('local')->delete($previous);
        }

        // R9: Store face images on private disk (not publicly accessible)
        $path = $validated['face_image']->store('face-search', 'local');
        $request->session()->forget($this->faceSessionKey($event));
        $request->session()->forget($this->faceSessionMatchesKey($event));

        $matches = [];
        try {
            $matches = $faceRecognition->match($event, Storage::disk('local')->path($path));
        } catch (\Throwable $exception) {
            report($exception);

            return redirect()->route('events.share', [
                'event' => $event->slug,
                'token' => $event->share_token,
            ])->with('face_search_error', 'Face recognition failed. Please try another photo.');
        }

        // Track face search
        app(\App\Services\AnalyticsService::class)->recordFaceSearch($event);

        $request->session()->put($this->faceSessionKey($event), true);
        $request->session()->put($this->faceSessionImageKey($event), $path);

        // FIX #8: Store timestamp with matched IDs for expiry
        $request->session()->put($this->faceSessionMatchesKey($event), [
            'matched_ids' => $matches,
            'matched_at' => now()->timestamp,
        ]);

        return redirect()->route('events.share', [
            'event' => $event->slug,
            'token' => $event->share_token,
        ]);
    }

    public function download(Request $request, Event $event, EventMedia $media)
    {
        if (! $this->tokenMatches($request, $event)) {
            abort(404);
        }

        if ($media->event_id !== $event->id) {
            abort(404);
        }

        if (! $event->allow_public_downloads) {
            abort(403);
        }

        if ($this->requiresPin($event) && ! $request->session()->get($this->pinSessionKey($event), false)) {
            abort(403);
        }

        // Track download
        app(\App\Services\AnalyticsService::class)->recordDownload($event, $media);

        $diskName = $media->disk;
        $path = $media->optimized_path;

        if (! $path || ! Storage::disk($diskName)->exists($path)) {
            $diskName = $media->originalDisk();
            $path = $media->original_path;
        }

        return Storage::disk($diskName)->download($path, $media->file_name);
    }

    public function storeGuestUpload(Request $request, Event $event)
    {
        if (! $this->tokenMatches($request, $event)) {
            abort(404);
        }

        if (! $event->allow_guest_upload) {
            abort(403, 'Guest uploads are not allowed for this event.');
        }

        // Check if event owner's plan allows guest uploads
        $event->load('createdBy');
        $owner = $event->createdBy;
        if ($owner) {
            $planLimit = app(\App\Services\PlanLimitService::class);
            if (! $planLimit->hasFeature($owner, 'has_guest_upload')) {
                abort(403, 'Guest uploads are not available on the current plan.');
            }
        }

        if ($this->requiresPin($event) && ! $request->session()->get($this->pinSessionKey($event), false)) {
            abort(403);
        }

        $request->validate([
            'files' => ['required', 'array', 'max:10'],
            'files.*' => ['file', 'mimes:jpg,jpeg,png,webp,mp4', 'max:51200'], // 50MB
        ]);

        // FIX #5: Validate guest upload against event owner's plan limits
        $event->load('createdBy.plan');
        $owner = $event->createdBy;

        if ($owner && $owner->plan) {
            // Check storage quota
            $usageService = app(\App\Services\StorageUsageService::class);

            try {
                foreach ($request->file('files') as $file) {
                    $usageService->assertCanStore((int) $file->getSize());
                }
            } catch (\App\Exceptions\StorageLimitReachedException $e) {
                return back()->with('error', 'Storage limit reached. The event owner needs to upgrade their plan.');
            }

            // Check file type limits (video uploads may be restricted)
            $planLimit = app(\App\Services\PlanLimitService::class);
            foreach ($request->file('files') as $file) {
                $mimeType = $file->getClientMimeType();
                $fileType = str_starts_with($mimeType, 'video/') ? 'video' : 'image';

                try {
                    $planLimit->assertCanUpload($event, $fileType, (int) $file->getSize());
                } catch (\App\Exceptions\PlanLimitReachedException $e) {
                    return back()->with('error', 'Upload limit reached. The event owner\'s plan does not allow this file type or size.');
                }
            }
        }

        // R10: Wrap guest upload in a transaction
        $uploaded = 0;
        if ($request->hasFile('files')) {
            DB::transaction(function () use ($request, $event, &$uploaded) {
                foreach ($request->file('files') as $file) {
                    $mime = $file->getMimeType();
                    $type = str_starts_with($mime, 'video/') ? 'video' : 'image';

                    $path = $file->store('events/' . $event->id . '/original', 'client_media');

                    $media = $event->media()->create([
                        'disk' => 'client_media',
                        'original_path' => $path,
                        'file_name' => $file->getClientOriginalName(),
                        'file_type' => $type,
                        'mime_type' => $mime,
                        'size' => $file->getSize(),
                        'status' => 'pending',
                        'is_guest_upload' => true,
                        'sort_order' => 0,
                    ]);
                    ProcessEventMedia::dispatch($media->id);
                    $uploaded++;
                }
            });
        }

        if ($request->wantsJson()) {
            return response()->json(['message' => "{$uploaded} files uploaded successfully."]);
        }

        return redirect()->route('events.share', [
            'event' => $event->slug,
            'token' => $event->share_token,
        ])->with('success', "{$uploaded} files uploaded successfully.");
    }

    // R9: Serve face search images from private disk
    public function faceImage(Request $request, Event $event)
    {
        if (! $this->tokenMatches($request, $event)) {
            abort(404);
        }

        $path = $request->session()->get($this->faceSessionImageKey($event));

        if (! $path || ! Storage::disk('local')->exists($path)) {
            abort(404);
        }

        return Storage::disk('local')->response($path);
    }

    private function shouldShowBranding(Event $event): bool
    {
        $owner = $event->createdBy ?? \App\Models\User::find($event->created_by);
        if (! $owner) {
            return true;
        }

        $planLimit = app(\App\Services\PlanLimitService::class);

        return ! $planLimit->hasFeature($owner, 'has_custom_branding');
    }

    private function tokenMatches(Request $request, Event $event): bool
    {
        $token = (string) $request->query('token', $request->input('token', ''));

        return $token !== '' && hash_equals($event->share_token ?? '', $token);
    }

    private function requiresPin(Event $event): bool
    {
        return (bool) ($event->guest_pin || $event->admin_pin);
    }

    /**
     * Check which PIN type matches. Returns 'admin', 'guest', or null.
     */
    private function pinMatches(Event $event, string $pin): ?string
    {
        foreach (['admin_pin' => 'admin', 'guest_pin' => 'guest'] as $field => $type) {
            $encrypted = $event->{$field};
            if (! $encrypted) {
                continue;
            }

            try {
                $decrypted = Crypt::decryptString($encrypted);
            } catch (\Throwable $exception) {
                continue;
            }

            if (hash_equals($decrypted, $pin)) {
                return $type;
            }
        }

        return null;
    }

    private function pinSessionKey(Event $event): string
    {
        return 'event_access.' . $event->id;
    }

    private function pinTypeSessionKey(Event $event): string
    {
        return 'event_access_type.' . $event->id;
    }

    private function faceSessionKey(Event $event): string
    {
        return 'event_face_search.' . $event->id;
    }

    private function faceSessionImageKey(Event $event): string
    {
        return 'event_face_image.' . $event->id;
    }

    private function faceSessionMatchesKey(Event $event): string
    {
        return 'event_face_matches.' . $event->id;
    }

    private function isExpired(Event $event): bool
    {
        if (! $event->expiry_date) {
            return false;
        }

        return $event->expiry_date->isPast();
    }

    private function emptyMediaStats(): array
    {
        return [
            'total' => 0,
            'ready' => 0,
            'processing' => 0,
            'pending' => 0,
        ];
    }
}
