<?php

namespace App\Http\Controllers;

use App\Models\EventMedia;
use App\Services\CloudStorageManager;
use App\Support\MediaObfuscator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class MediaPreviewController extends Controller
{
    public function show(Request $request, EventMedia $media)
    {
        app(CloudStorageManager::class)->registerDisks();

        $variant = $request->query('variant', 'preview');

        [$diskName, $path] = match ($variant) {
            'thumbnail' => [$media->disk, $media->thumbnail_path],
            'original' => [$media->originalDisk(), $media->original_path],
            default => $this->resolvePreviewDisk($media),
        };

        if (! $path || ! Storage::disk($diskName)->exists($path)) {
            abort(404);
        }

        // Obfuscated files need streaming with header stripped
        $isObfuscated = $media->meta['obfuscated'] ?? false;
        if ($isObfuscated) {
            return MediaObfuscator::streamResponse(
                $diskName,
                $path,
                $media->file_name,
                $media->mime_type
            );
        }

        // Non-obfuscated (pending or legacy) — serve normally
        return Storage::disk($diskName)->response($path, $media->file_name, [
            'Cache-Control' => 'private, max-age=600',
        ]);
    }

    private function resolvePreviewDisk(EventMedia $media): array
    {
        $path = $media->optimized_path ?: $media->original_path;

        // When optimized_path == original_path, file lives on the original disk
        if ($media->optimized_path && $media->optimized_path === $media->original_path) {
            return [$media->originalDisk(), $path];
        }

        return [$media->disk, $path];
    }
}
