<?php

namespace App\Http\Controllers;

use App\Models\UserDeleteRequest;
use App\Http\Requests\ProfileUpdateRequest;
use App\Support\ActivityLogger;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;
use Illuminate\View\View;

class ProfileController extends Controller
{
    /**
     * Display the user's profile form.
     */
    public function edit(Request $request): View
    {
        return view('profile.edit', [
            'user' => $request->user(),
            'deleteRequest' => UserDeleteRequest::query()
                ->where('user_id', $request->user()->id)
                ->latest('id')
                ->first(),
        ]);
    }

    /**
     * Update the user's profile information.
     */
    public function update(ProfileUpdateRequest $request): RedirectResponse
    {
        $request->user()->fill($request->validated());

        if ($request->user()->isDirty('email')) {
            $request->user()->email_verified_at = null;
        }

        $request->user()->save();

        return Redirect::route('profile.edit')->with('status', 'profile-updated');
    }

    /**
     * Delete the user's account.
     */
    public function destroy(Request $request): RedirectResponse
    {
        $request->validateWithBag('userDeletion', [
            'password' => ['required', 'current_password'],
        ]);

        $user = $request->user();

        $pending = UserDeleteRequest::query()
            ->where('user_id', $user->id)
            ->where('status', UserDeleteRequest::STATUS_PENDING)
            ->first();

        if ($pending) {
            return Redirect::route('profile.edit')
                ->with('status', 'Your delete account request is already pending review.');
        }

        $deleteRequest = UserDeleteRequest::create([
            'user_id' => $user->id,
            'user_name' => $user->name,
            'user_email' => $user->email,
            'status' => UserDeleteRequest::STATUS_PENDING,
        ]);

        ActivityLogger::logFromRequest(
            $request,
            'create',
            UserDeleteRequest::class,
            $deleteRequest->id,
            'Delete account requested by ' . ($user->name ?: $user->email)
        );

        return Redirect::route('profile.edit')
            ->with('status', 'Delete account request submitted. An admin will review it shortly.');
    }
}
