<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class EnsureAdminPermission
{
    private const PERMISSION_MODULES = [
        'activity-logs',
        'blocked-ips',
        'blocked-locations',
        'blog-categories',
        'blog-posts',
        'cities',
        'countries',
        'coupons',
        'currencies',
        'events',
        'extensions',
        'fact-stats',
        'faqs',
        'languages',
        'pages',
        'payment-gateways',
        'permissions',
        'plans',
        'reports',
        'roles',
        'security-audit-logs',
        'security-settings',
        'states',
        'support-tickets',
        'system-logs',
        'system-settings',
        'testimonials',
        'ticket-assignments',
        'users',
        'user-delete-requests',
    ];

    private const ACTION_MAP = [
        'index' => 'view',
        'show' => 'view',
        'create' => 'create',
        'store' => 'create',
        'edit' => 'update',
        'update' => 'update',
        'destroy' => 'delete',
        'bulk-destroy' => 'delete',
        'export' => 'export',
        'import' => 'import',
        'download' => 'view',
        'notifications' => 'view',
    ];

    public function handle(Request $request, Closure $next): Response
    {
        $route = $request->route();
        if (! $route) {
            return $next($request);
        }

        $name = $route->getName();
        if (! $name || ! str_starts_with($name, 'admin.')) {
            return $next($request);
        }

        $user = $request->user();
        if (! $user) {
            abort(401);
        }

        $segments = explode('.', $name);
        $module = $segments[1] ?? null;
        if (! $module || ! in_array($module, self::PERMISSION_MODULES, true)) {
            return $next($request);
        }

        $actionSegment = (string) end($segments);
        $permission = $this->resolvePermission($module, $segments, $actionSegment);

        if (! $permission) {
            return $next($request);
        }

        if (! $user->can($permission)) {
            abort(403);
        }

        return $next($request);
    }

    private function resolvePermission(string $module, array $segments, string $actionSegment): ?string
    {
        if ($module === 'events' && in_array('media', $segments, true)) {
            $viewActions = ['download', 'export', 'bulk-download'];
            $action = in_array($actionSegment, $viewActions, true) ? 'view' : 'update';

            return $module . '.' . $action;
        }

        if ($module === 'reports') {
            $action = $actionSegment === 'export' ? 'export' : 'view';

            return $module . '.' . $action;
        }

        if (in_array($module, ['system-settings', 'security-settings', 'extensions', 'payment-gateways'], true)) {
            $action = $actionSegment === 'index' ? 'view' : 'update';

            return $module . '.' . $action;
        }

        $action = self::ACTION_MAP[$actionSegment] ?? null;

        return $action ? $module . '.' . $action : null;
    }
}
