<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class EnsureApiPermission
{
    /**
     * Map of API route prefixes to permission module names.
     */
    private const ROUTE_MODULE_MAP = [
        'users' => 'users',
        'roles' => 'roles',
        'permissions' => 'permissions',
        'countries' => 'countries',
        'states' => 'states',
        'cities' => 'cities',
        'currencies' => 'currencies',
        'languages' => 'languages',
        'events' => 'events',
        'plans' => 'plans',
        'coupons' => 'coupons',
        'support-tickets' => 'support-tickets',
        'faqs' => 'faqs',
        'pages' => 'pages',
        'testimonials' => 'testimonials',
        'fact-stats' => 'fact-stats',
        'blog-categories' => 'blog-categories',
        'blog-posts' => 'blog-posts',
        'ticket-assignments' => 'ticket-assignments',
        'blocked-ips' => 'blocked-ips',
        'blocked-locations' => 'blocked-locations',
        'system-logs' => 'system-logs',
        'activity-logs' => 'activity-logs',
        'security-audit-logs' => 'security-audit-logs',
        'reports' => 'reports',
    ];

    /**
     * Map of HTTP methods to permission actions.
     */
    private const METHOD_ACTION_MAP = [
        'GET' => 'view',
        'HEAD' => 'view',
        'POST' => 'create',
        'PUT' => 'update',
        'PATCH' => 'update',
        'DELETE' => 'delete',
    ];

    public function handle(Request $request, Closure $next): Response
    {
        $user = $request->user();

        if (! $user) {
            abort(401, 'Unauthenticated.');
        }

        $path = $request->decodedPath();

        // Strip the 'api/' and version prefixes if present
        $path = preg_replace('#^api/(v\d+/)?#', '', $path);

        $module = $this->resolveModule($path);

        if (! $module) {
            return $next($request);
        }

        $action = $this->resolveAction($request, $path);

        $permission = $module . '.' . $action;

        if (! $user->can($permission)) {
            abort(403, 'You do not have permission to perform this action.');
        }

        return $next($request);
    }

    private function resolveModule(string $path): ?string
    {
        foreach (self::ROUTE_MODULE_MAP as $prefix => $module) {
            if ($path === $prefix || str_starts_with($path, $prefix . '/')) {
                return $module;
            }
        }

        return null;
    }

    private function resolveAction(Request $request, string $path): string
    {
        if (str_ends_with($path, '/export')) {
            return 'export';
        }

        if (str_ends_with($path, '/import')) {
            return 'import';
        }

        if (str_ends_with($path, '/bulk')) {
            return 'delete';
        }

        $method = strtoupper($request->method());

        return self::METHOD_ACTION_MAP[$method] ?? 'view';
    }
}
