<?php

namespace App\Http\Middleware;

use App\Models\BlockedIp;
use App\Support\SecurityAuditLogger;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Schema;
use Symfony\Component\HttpFoundation\IpUtils;

class EnsureIpNotBlocked
{
    public function handle(Request $request, Closure $next)
    {
        if (app()->runningInConsole()) {
            return $next($request);
        }

        try {
            if (! Schema::hasTable('blocked_ips')) {
                return $next($request);
            }
        } catch (\Exception $e) {
            return $next($request);
        }

        $ipAddress = $request->ip();

        if (! $ipAddress) {
            return $next($request);
        }

        // R6: Cache blocked IPs for 5 minutes to avoid DB query on every request
        $blockedIps = Cache::remember('blocked_ips_active', 300, function () {
            return BlockedIp::query()
                ->where('is_active', true)
                ->get(['id', 'ip_address', 'reason'])
                ->toArray();
        });

        foreach ($blockedIps as $blocked) {
            try {
                if (IpUtils::checkIp($ipAddress, $blocked['ip_address'])) {
                    SecurityAuditLogger::log('ip_blocked', 'blocked', $request->user(), $request, [
                        'blocked_ip_id' => $blocked['id'],
                        'blocked_ip' => $blocked['ip_address'],
                        'reason' => $blocked['reason'] ?? null,
                    ]);

                    abort(403, 'Access denied.');
                }
            } catch (\Throwable $e) {
                continue;
            }
        }

        return $next($request);
    }
}
