<?php

namespace App\Http\Middleware;

use App\Models\BlockedLocation;
use App\Models\Country;
use App\Support\SecurityAuditLogger;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;

class EnsureLocationNotBlocked
{
    public function handle(Request $request, Closure $next)
    {
        if (app()->runningInConsole()) {
            return $next($request);
        }

        try {
            if (! Schema::hasTable('blocked_locations')) {
                return $next($request);
            }
        } catch (\Exception $e) {
            return $next($request);
        }

        $user = $request->user();
        $countryId = $user?->country_id;
        $stateId = $user?->state_id;
        $cityId = $user?->city_id;

        if (! $countryId) {
            $countryCode = $request->header('CF-IPCountry') ?: $request->header('X-Country-Code');
            $hasCountriesTable = false;
            try { $hasCountriesTable = Schema::hasTable('countries'); } catch (\Exception $e) {}
            if ($countryCode && $hasCountriesTable) {
                $countryId = Country::query()
                    ->where('iso2', strtoupper($countryCode))
                    ->value('id');
            }
        }

        if (! $countryId && ! $stateId && ! $cityId) {
            return $next($request);
        }

        $blockedLocations = BlockedLocation::query()->where('is_active', true)->get();

        foreach ($blockedLocations as $blocked) {
            if ($blocked->country_id && (! $countryId || (int) $blocked->country_id !== (int) $countryId)) {
                continue;
            }
            if ($blocked->state_id && (! $stateId || (int) $blocked->state_id !== (int) $stateId)) {
                continue;
            }
            if ($blocked->city_id && (! $cityId || (int) $blocked->city_id !== (int) $cityId)) {
                continue;
            }

            SecurityAuditLogger::log('location_blocked', 'blocked', $user, $request, [
                'blocked_location_id' => $blocked->id,
                'country_id' => $blocked->country_id,
                'state_id' => $blocked->state_id,
                'city_id' => $blocked->city_id,
                'reason' => $blocked->reason,
            ]);

            abort(403, 'Access denied.');
        }

        return $next($request);
    }
}
