<?php

namespace App\Http\Middleware;

use App\Models\SystemSetting;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;
use Symfony\Component\HttpFoundation\Response;

class EnsureMaintenanceModeApi
{
    public function handle(Request $request, Closure $next): Response
    {
        if (app()->runningInConsole()) {
            return $next($request);
        }

        try {
            if (! Schema::hasTable('system_settings')) {
                return $next($request);
            }
        } catch (\Exception $e) {
            return $next($request);
        }

        $payload = SystemSetting::query()->value('payload');

        if (is_string($payload)) {
            $decoded = json_decode($payload, true);
            $payload = is_array($decoded) ? $decoded : [];
        }

        $maintenance = is_array($payload) ? ($payload['maintenance'] ?? []) : [];
        $enabled = filter_var($maintenance['maintenance_enabled'] ?? false, FILTER_VALIDATE_BOOLEAN);

        if (! $enabled) {
            return $next($request);
        }

        if ($this->shouldBypass($request)) {
            return $next($request);
        }

        $message = trim((string) ($maintenance['maintenance_message'] ?? ''));
        $message = $message !== '' ? $message : 'We are performing scheduled maintenance. Please check back shortly.';

        return response()->json([
            'data' => null,
            'meta' => ['maintenance' => true],
            'errors' => [['code' => '503', 'message' => $message]],
        ], 503);
    }

    private function shouldBypass(Request $request): bool
    {
        $user = $request->user();

        if ($user && method_exists($user, 'hasAnyRole')) {
            $adminRoles = config('security.admin_roles', []);

            if ($adminRoles && $user->hasAnyRole($adminRoles)) {
                return true;
            }
        }

        return false;
    }
}
