<?php

namespace App\Http\Middleware;

use App\Support\ActivityLogger;
use Closure;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Symfony\Component\HttpFoundation\Response;

class LogAdminActivity
{
    public function handle(Request $request, Closure $next): Response
    {
        $response = $next($request);

        if (app()->runningInConsole()) {
            return $response;
        }

        $user = $request->user();
        if (! $user) {
            return $response;
        }

        $route = $request->route();
        $routeName = $route?->getName();
        if (! $routeName || ! Str::startsWith($routeName, 'admin.')) {
            return $response;
        }

        if ($this->shouldSkipRoute($routeName)) {
            return $response;
        }

        if ($request->session()->has('errors') && $request->session()->getOldInput()) {
            return $response;
        }

        $action = $this->resolveAction($request, $routeName);
        if (! $action) {
            return $response;
        }

        if ($response->getStatusCode() >= 400) {
            return $response;
        }

        [$subjectType, $subjectId, $subjectLabel] = $this->resolveSubject($routeName, $route?->parameters() ?? []);
        $description = $this->buildDescription($action, $subjectType, $subjectId, $subjectLabel, $request, $routeName);

        ActivityLogger::logFromRequest($request, $action, $subjectType, $subjectId, $description);

        return $response;
    }

    private function shouldSkipRoute(string $routeName): bool
    {
        return Str::startsWith($routeName, [
            'admin.activity-logs.',
            'admin.system-logs.',
            'admin.notifications',
        ]);
    }

    private function resolveAction(Request $request, string $routeName): ?string
    {
        if (Str::contains($routeName, '.test')) {
            return 'other';
        }

        if (Str::contains($routeName, ['export', '.download'])) {
            return 'export';
        }

        if (Str::contains($routeName, 'import')) {
            return 'import';
        }

        if (Str::contains($routeName, 'bulk-destroy') || Str::endsWith($routeName, '.destroy')) {
            return 'delete';
        }

        if (Str::endsWith($routeName, '.update') || Str::contains($routeName, ['.toggle', '.cover', '.featured'])) {
            return 'update';
        }

        if (in_array($request->method(), ['PUT', 'PATCH'], true)) {
            return 'update';
        }

        if (Str::endsWith($routeName, '.store') || $request->isMethod('post')) {
            return 'create';
        }

        return null;
    }

    private function resolveSubject(string $routeName, array $parameters): array
    {
        foreach ($parameters as $param) {
            if ($param instanceof Model) {
                return [class_basename($param), (int) $param->getKey(), $this->resolveModelLabel($param)];
            }
        }

        $subjectType = $this->subjectFromRoute($routeName);

        return [$subjectType, null, null];
    }

    private function subjectFromRoute(string $routeName): ?string
    {
        if (Str::contains($routeName, '.media.')) {
            return 'Event Media';
        }

        $resource = Str::before(Str::after($routeName, 'admin.'), '.');

        $map = [
            'users' => 'User',
            'roles' => 'Role',
            'permissions' => 'Permission',
            'events' => 'Event',
            'plans' => 'Plan',
            'coupons' => 'Coupon',
            'support-tickets' => 'Support Ticket',
            'ticket-assignments' => 'Ticket Assignment',
            'faqs' => 'FAQ',
            'system-settings' => 'System Settings',
            'extensions' => 'Extensions',
            'security-settings' => 'Security Settings',
            'pages' => 'Page',
            'system-logs' => 'System Log',
            'activity-logs' => 'Activity Log',
            'countries' => 'Country',
            'states' => 'State',
            'cities' => 'City',
            'currencies' => 'Currency',
            'languages' => 'Language',
            'testimonials' => 'Testimonial',
            'fact-stats' => 'Fact Stat',
            'blog-categories' => 'Blog Category',
            'blog-posts' => 'Blog Post',
            'reports' => 'Report',
            'blocked-ips' => 'Blocked IP',
            'blocked-locations' => 'Blocked Location',
            'security-audit-logs' => 'Security Audit Log',
            'locale' => 'Locale',
            'user-delete-requests' => 'User Delete Request',
        ];

        if (isset($map[$resource])) {
            return $map[$resource];
        }

        $resource = trim($resource);
        if ($resource === '') {
            return null;
        }

        return Str::title(str_replace('-', ' ', $resource));
    }

    private function buildDescription(
        string $action,
        ?string $subjectType,
        ?int $subjectId,
        ?string $subjectLabel,
        Request $request,
        string $routeName
    ): string {
        $label = $subjectType ?: 'Record';
        $labelSuffix = '';

        if ($subjectId) {
            $labelSuffix .= ' #' . $subjectId;
        }

        if ($subjectLabel) {
            $labelSuffix .= ' "' . $subjectLabel . '"';
        }

        if ($action === 'export') {
            $reportKey = $request->route('report');
            $format = $request->route('format');
            if ($reportKey) {
                return $this->formatAction($action) . " Report {$reportKey}" . ($format ? " ({$format})" : '');
            }

            return $this->formatAction($action) . ' ' . $label;
        }

        if ($action === 'import') {
            return $this->formatAction($action) . ' ' . Str::plural($label);
        }

        if ($action === 'delete') {
            $ids = $request->input('ids');
            if (is_array($ids) && count($ids) > 1) {
                return $this->formatAction($action) . ' ' . count($ids) . ' ' . Str::plural($label);
            }
        }

        if ($action === 'update' && $subjectType === 'User Delete Request') {
            $decision = strtolower((string) $request->input('action'));
            if (in_array($decision, ['accept', 'reject'], true)) {
                $statusLabel = $decision === 'accept' ? 'Accepted' : 'Rejected';

                return 'User delete request ' . $statusLabel . $labelSuffix;
            }
        }

        return $this->formatAction($action) . ' ' . $label . $labelSuffix;
    }

    private function formatAction(string $action): string
    {
        return Str::ucfirst($action);
    }

    private function resolveModelLabel(Model $model): ?string
    {
        foreach (['name', 'title', 'email', 'slug'] as $field) {
            $value = $model->getAttribute($field);
            if (is_string($value) && trim($value) !== '') {
                return Str::limit(trim($value), 40, '...');
            }
        }

        return null;
    }
}
