<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Symfony\Component\HttpFoundation\Response;

/**
 * Web Cron — runs the Laravel scheduler and queue worker AFTER the
 * response is sent. On LiteSpeed / PHP-FPM / Nginx this executes
 * silently in the background so the admin never notices any delay.
 *
 * Handles ALL scheduled tasks automatically:
 *  - Queue worker (process media, face recognition, imports)
 *  - FTP camera upload ingestion (every 2 min)
 *  - Daily maintenance (cleanup deleted media, GDPR face data)
 *
 * No cron jobs, no terminal commands needed. Everything runs
 * automatically when any admin browses the panel.
 */
class ProcessQueueOnRequest
{
    public function handle(Request $request, Closure $next): Response
    {
        return $next($request);
    }

    /**
     * Runs after the response has been sent to the client.
     */
    public function terminate(Request $request, Response $response): void
    {
        // Skip AJAX / JSON requests — the AJAX auto-processor handles those
        if ($request->ajax() || $request->expectsJson()) {
            return;
        }

        $this->runScheduler();
        $this->runDailyMaintenance();
    }

    /**
     * Run the Laravel scheduler (queue:work, FTP ingest, etc.)
     * once per 60 seconds via file lock.
     */
    private function runScheduler(): void
    {
        $lockFile = storage_path('framework/webcron.lock');

        $lastRun = 0;
        if (file_exists($lockFile)) {
            $lastRun = (int) file_get_contents($lockFile);
        }

        if (time() - $lastRun < 60) {
            return;
        }

        // Claim the lock first
        file_put_contents($lockFile, (string) time());

        // Run the full scheduler — this handles:
        // - queue:work (every minute)
        // - events:ingest-ftp-all (every 2 min)
        // - media:cleanup-deleted (daily at 2 AM — when admin is online)
        // - gdpr:cleanup-face-data (daily at 3 AM — when admin is online)
        try {
            Artisan::call('schedule:run');
        } catch (\Throwable $e) {
            \Log::warning('[WebCron] Scheduler error: ' . $e->getMessage());
        }

        // Also directly process queue if there are pending jobs
        // (schedule:run may run queue:work in background which doesn't
        //  always work on shared hosting without pcntl)
        try {
            $pending = DB::table('jobs')->count();
            if ($pending > 0) {
                Artisan::call('queue:work', [
                    '--stop-when-empty' => true,
                    '--max-time'        => 55,
                    '--tries'           => 3,
                    '--memory'          => 256,
                    '--quiet'           => true,
                ]);
            }
        } catch (\Throwable $e) {
            \Log::warning('[WebCron] Queue processing error: ' . $e->getMessage());
        }
    }

    /**
     * Run daily maintenance tasks if they haven't run in 24+ hours.
     * This catches cleanup tasks that the scheduler misses because
     * no admin was online at the exact scheduled time (2-3 AM).
     */
    private function runDailyMaintenance(): void
    {
        $lockFile = storage_path('framework/webcron-daily.lock');

        $lastRun = 0;
        if (file_exists($lockFile)) {
            $lastRun = (int) file_get_contents($lockFile);
        }

        // Only run once per 24 hours
        if (time() - $lastRun < 86400) {
            return;
        }

        file_put_contents($lockFile, (string) time());

        // Cleanup soft-deleted media older than 30 days
        try {
            Artisan::call('media:cleanup-deleted', ['--force' => true]);
        } catch (\Throwable $e) {
            \Log::warning('[WebCron] Media cleanup error: ' . $e->getMessage());
        }

        // GDPR: Remove face data for expired events
        try {
            Artisan::call('gdpr:cleanup-face-data');
        } catch (\Throwable $e) {
            \Log::warning('[WebCron] GDPR cleanup error: ' . $e->getMessage());
        }
    }
}
