<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;

class SecurityHeaders
{
    /**
     * Payment gateway domains that require external script/frame/connect access.
     * Using wildcard subdomains (*.domain.com) for robustness since payment
     * SDKs often load resources from multiple subdomains.
     */
    private const PAYMENT_DOMAINS = [
        'https://*.razorpay.com',       // Razorpay checkout, API, telemetry
        'https://*.cashfree.com',       // Cashfree SDK
        'https://*.braintreegateway.com', // Braintree drop-in UI
        'https://*.paypal.com',         // PayPal buttons, checkout
        'https://*.midtrans.com',       // Midtrans Snap
        'https://*.oppwa.com',          // HyperPay widget
        'https://*.stripe.com',         // Stripe.js (if ever used)
        'https://*.squareup.com',       // Square Web SDK
        'https://*.squareupsandbox.com', // Square sandbox
    ];

    public function handle(Request $request, Closure $next)
    {
        $response = $next($request);

        $response->headers->set('X-Content-Type-Options', 'nosniff');
        $response->headers->set('X-Frame-Options', 'SAMEORIGIN');
        $response->headers->set('Referrer-Policy', 'strict-origin-when-cross-origin');

        // Allow camera only on event share/face-search pages; block everywhere else
        $needsCamera = $request->routeIs('events.share', 'events.share.*');
        $cameraPolicy = $needsCamera ? 'camera=self' : 'camera=()';
        $response->headers->set('Permissions-Policy', "{$cameraPolicy}, microphone=(), geolocation=()");

        $paymentDomains = implode(' ', self::PAYMENT_DOMAINS);

        $response->headers->set(
            'Content-Security-Policy',
            "default-src 'self'; "
            . "script-src 'self' 'unsafe-inline' 'unsafe-eval' https://www.google.com https://www.gstatic.com https://cdn.jsdelivr.net {$paymentDomains}; "
            . "style-src 'self' 'unsafe-inline' https://fonts.googleapis.com https://fonts.bunny.net {$paymentDomains}; "
            . "font-src 'self' https://fonts.gstatic.com https://fonts.bunny.net; "
            . "img-src 'self' data: blob: https:; "
            . "frame-src 'self' https://www.google.com {$paymentDomains}; "
            . "connect-src 'self' {$paymentDomains};"
        );

        return $response;
    }
}
