<?php

namespace App\Http\Requests\Admin;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class UpdateSystemSettingsRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function withValidator($validator): void
    {
        $validator->after(function ($validator) {
            $file = $this->file('watermark_image');
            if (! $file || $file->isValid()) {
                return;
            }

            $errors = $validator->errors();
            $errors->forget('watermark_image');
            $errors->add('watermark_image', 'Upload failed (' . $file->getError() . '): ' . $file->getErrorMessage());
        });
    }

    public function rules(): array
    {
        $group = $this->route('group');

        return match ($group) {
            'general' => [
                'default_language' => ['nullable', 'string', 'max:50', Rule::exists('languages', 'code')],
                'default_currency' => ['nullable', 'string', 'max:10', Rule::exists('currencies', 'code')],
                'default_timezone' => ['nullable', 'string', 'max:64'],
                'date_format' => ['nullable', 'string', 'max:32'],
                'time_format' => ['nullable', 'string', 'max:32'],
                'support_email' => ['nullable', 'email', 'max:255'],
                'support_phone' => ['nullable', 'string', 'max:20'],
                'enable_multi_language' => ['sometimes', 'boolean'],
            ],
            'site' => [
                'site_name' => ['nullable', 'string', 'max:120'],
                'site_logo' => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp', 'max:5120'],
                'favicon' => ['nullable', 'file', 'mimes:jpg,jpeg,png,webp,ico', 'max:2048'],
                'admin_logo' => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp', 'max:5120'],
                'company_name' => ['nullable', 'string', 'max:120'],
                'company_address' => ['nullable', 'string', 'max:255'],
                'company_country_id' => ['nullable', 'integer', 'exists:countries,id'],
                'company_state_id' => ['nullable', 'integer', 'exists:states,id'],
                'company_city_id' => ['nullable', 'integer', 'exists:cities,id'],
                'company_phone' => ['nullable', 'string', 'max:20'],
                'business_hours' => ['nullable', 'string', 'max:120'],
                'latitude' => ['nullable', 'numeric', 'between:-90,90'],
                'longitude' => ['nullable', 'numeric', 'between:-180,180'],
                'right_click_disable' => ['sometimes', 'boolean'],
                'uc_browser_block' => ['sometimes', 'boolean'],
            ],
            'smtp' => [
                'smtp_host' => ['nullable', 'string', 'max:120'],
                'smtp_port' => ['nullable', 'integer', 'min:1', 'max:65535'],
                'smtp_username' => ['nullable', 'string', 'max:120'],
                'smtp_password' => ['nullable', 'string', 'max:255'],
                'smtp_encryption' => ['nullable', 'in:tls,ssl,none'],
                'smtp_from_address' => ['nullable', 'email', 'max:255'],
                'smtp_from_name' => ['nullable', 'string', 'max:120'],
            ],
            'footer' => [
                'footer_logo' => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp', 'max:5120'],
                'footer_social' => ['nullable', 'array'],
                'footer_social.*' => ['nullable', 'url', 'max:255'],
                'app_links' => ['nullable', 'array'],
                'app_links.*' => ['nullable', 'url', 'max:255'],
            ],
            'seo' => [
                'meta_title' => ['nullable', 'string', 'max:120'],
                'meta_description' => ['nullable', 'string', 'max:500'],
                'meta_keywords' => ['nullable', 'string', 'max:255'],
                'sitemap_enabled' => ['sometimes', 'boolean'],
                'robots_txt' => ['nullable', 'string'],
                'llm_txt' => ['nullable', 'string'],
            ],
            'social' => [
                'providers' => ['nullable', 'array'],
                'providers.*.enabled' => ['nullable', 'boolean'],
                'providers.*.client_id' => ['nullable', 'string', 'max:255'],
                'providers.*.client_secret' => ['nullable', 'string', 'max:255'],
                'providers.*.redirect_url' => ['nullable', 'url', 'max:255'],
            ],
            'gdpr' => [
                'gdpr_enabled' => ['sometimes', 'boolean'],
                'gdpr_message' => ['nullable', 'string', 'max:500'],
                'gdpr_accept_label' => ['nullable', 'string', 'max:60'],
                'gdpr_decline_label' => ['nullable', 'string', 'max:60'],
            ],
            'media' => [
                'media_max_file_size' => ['nullable', 'numeric', 'min:1', 'max:2048'],
                'max_file_size' => ['nullable', 'numeric', 'min:1', 'max:2048'],
                'chunk_size' => ['nullable', 'numeric', 'min:1', 'max:512'],
                'max_width' => ['nullable', 'integer', 'min:600', 'max:8000'],
                'quality' => ['nullable', 'integer', 'min:40', 'max:100'],
                'storage_limit_gb' => ['nullable', 'numeric', 'min:0', 'max:1000'],
                'ftp_path' => ['nullable', 'string', 'max:255'],
                'watermark_enabled' => ['sometimes', 'boolean'],
                'watermark_image' => [
                    'nullable',
                    'image',
                    'mimes:jpg,jpeg,png,webp',
                    'max:5120',
                    'dimensions:max_width=2000,max_height=2000', // FIX #15: Prevent corrupt/huge watermarks
                ],
                'watermark_position' => ['nullable', Rule::in(array_keys(config('events.media.watermark.positions', [])))],
            ],
            'maintenance' => [
                'maintenance_enabled' => ['sometimes', 'boolean'],
                'maintenance_message' => ['nullable', 'string', 'max:500'],
            ],
            'cloud_storage' => [
                'enabled' => ['sometimes', 'boolean'],
                'provider' => ['required', 'in:s3,gcs,gdrive,azure,bunny,wasabi'],
                'visibility' => ['nullable', 'in:public,private'],
                'base_path' => ['nullable', 'string', 'max:255'],
                'limit_enabled' => ['sometimes', 'boolean'],
                'limit_value' => ['nullable', 'numeric', 'min:0', 'max:100000'],
                'limit_unit' => ['nullable', 'in:gb,tb'],
                'providers' => ['nullable', 'array'],
                'providers.s3.key' => ['nullable', 'string', 'max:255'],
                'providers.s3.secret' => ['nullable', 'string', 'max:255'],
                'providers.s3.region' => ['nullable', 'string', 'max:64'],
                'providers.s3.bucket' => ['nullable', 'string', 'max:255'],
                'providers.s3.endpoint' => ['nullable', 'string', 'max:255'],
                'providers.s3.url' => ['nullable', 'string', 'max:255'],
                'providers.s3.use_path_style_endpoint' => ['nullable', 'boolean'],
                'providers.gcs.project_id' => ['nullable', 'string', 'max:255'],
                'providers.gcs.bucket' => ['nullable', 'string', 'max:255'],
                'providers.gcs.key_file' => ['nullable', 'string', 'max:20000'],
                'providers.gdrive.client_id' => ['nullable', 'string', 'max:255'],
                'providers.gdrive.client_secret' => ['nullable', 'string', 'max:255'],
                'providers.gdrive.refresh_token' => ['nullable', 'string', 'max:2000'],
                'providers.gdrive.folder' => ['nullable', 'string', 'max:255'],
                'providers.gdrive.team_drive_id' => ['nullable', 'string', 'max:255'],
                'providers.gdrive.shared_folder_id' => ['nullable', 'string', 'max:255'],
                'providers.azure.account_name' => ['nullable', 'string', 'max:255'],
                'providers.azure.account_key' => ['nullable', 'string', 'max:255'],
                'providers.azure.container' => ['nullable', 'string', 'max:255'],
                'providers.azure.endpoint' => ['nullable', 'string', 'max:255'],
                'providers.azure.url' => ['nullable', 'string', 'max:255'],
                'providers.bunny.storage_zone' => ['nullable', 'string', 'max:255'],
                'providers.bunny.access_key' => ['nullable', 'string', 'max:255'],
                'providers.bunny.region' => ['nullable', 'string', 'max:64'],
                'providers.bunny.endpoint' => ['nullable', 'string', 'max:255'],
                'providers.bunny.url' => ['nullable', 'string', 'max:255'],
                'providers.wasabi.key' => ['nullable', 'string', 'max:255'],
                'providers.wasabi.secret' => ['nullable', 'string', 'max:255'],
                'providers.wasabi.region' => ['nullable', 'string', 'max:64'],
                'providers.wasabi.bucket' => ['nullable', 'string', 'max:255'],
                'providers.wasabi.endpoint' => ['nullable', 'string', 'max:255'],
                'providers.wasabi.url' => ['nullable', 'string', 'max:255'],
                'providers.wasabi.use_path_style_endpoint' => ['nullable', 'boolean'],
            ],
            'pwa' => [
                'pwa_enabled'      => ['sometimes', 'boolean'],
                'app_name'         => ['nullable', 'string', 'max:120'],
                'app_url'          => ['nullable', 'string', 'max:255'],
                'icon'             => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp', 'max:5120'],
                'splash_screen'    => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp', 'max:5120'],
                'theme_color'      => ['nullable', 'string', 'max:20'],
                'background_color' => ['nullable', 'string', 'max:20'],
            ],
            default => [],
        };
    }
}
