<?php

namespace App\Http\Requests\Auth;

use App\Models\User;
use App\Support\SecurityAuditLogger;
use App\Support\SecuritySettings;
use Illuminate\Auth\Events\Lockout;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class LoginRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $rules = [
            'email' => ['required', 'string', 'email'],
            'password' => ['required', 'string'],
        ];

        $captchaEnabled = SecuritySettings::getValue('captcha_enabled', false);
        if ($captchaEnabled) {
            $rules['g-recaptcha-response'] = ['nullable', 'string'];
        }

        return $rules;
    }

    /**
     * Attempt to authenticate the request's credentials.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function authenticate(): void
    {
        $this->ensureIsNotRateLimited();

        // $this->ensureCaptchaIsValid();

        if (! Auth::attempt($this->only('email', 'password'), $this->boolean('remember'))) {
            RateLimiter::hit($this->throttleKey(), $this->decaySeconds());

            $user = $this->resolveUser();
            SecurityAuditLogger::log('login_failed', 'failed', $user, $this, [
                'email' => $this->string('email')->toString(),
                'attempts' => RateLimiter::attempts($this->throttleKey()),
            ]);

            throw ValidationException::withMessages([
                'email' => trans('auth.failed'),
            ]);
        }

        RateLimiter::clear($this->throttleKey());
    }

    /**
     * Ensure the login request is not rate limited.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function ensureIsNotRateLimited(): void
    {
        if (! RateLimiter::tooManyAttempts($this->throttleKey(), $this->maxAttempts())) {
            return;
        }

        event(new Lockout($this));

        $seconds = RateLimiter::availableIn($this->throttleKey());
        $user = $this->resolveUser();
        SecurityAuditLogger::log('login_throttled', 'blocked', $user, $this, [
            'email' => $this->string('email')->toString(),
            'seconds_remaining' => $seconds,
        ]);

        // Send lockout notification email
        if ($user && $user->email) {
            try {
                Mail::raw(
                    "Your account has been temporarily locked due to too many failed login attempts.\n\n" .
                    "IP Address: {$this->ip()}\n" .
                    "Time: " . now()->toDateTimeString() . "\n" .
                    "Lockout duration: " . ceil($seconds / 60) . " minute(s)\n\n" .
                    "If this wasn't you, please change your password immediately.",
                    function ($message) use ($user) {
                        $message->to($user->email)
                            ->subject('Account Lockout Alert - Suspicious Login Activity');
                    }
                );
            } catch (\Throwable $e) {
                Log::warning('Failed to send lockout notification', ['email' => $user->email]);
            }
        }

        throw ValidationException::withMessages([
            'email' => trans('auth.throttle', [
                'seconds' => $seconds,
                'minutes' => ceil($seconds / 60),
            ]),
        ]);
    }

    /**
     * Get the rate limiting throttle key for the request.
     */
    public function throttleKey(): string
    {
        return Str::transliterate(Str::lower($this->string('email')).'|'.$this->ip());
    }

    private function ensureCaptchaIsValid(): void
    {
        $captchaEnabled = SecuritySettings::getValue('captcha_enabled', false);

        if (! $captchaEnabled) {
            return;
        }

        $secret = (string) SecuritySettings::getValue('captcha_secret_key', '');
        if (trim($secret) === '') {
            throw ValidationException::withMessages([
                'g-recaptcha-response' => 'Captcha is enabled but not configured.',
            ]);
        }

        $response = (string) $this->input('g-recaptcha-response');

        if ($response === '') {
            throw ValidationException::withMessages([
                'g-recaptcha-response' => 'Captcha verification failed.',
            ]);
        }

        $verification = Http::asForm()
            ->timeout(5)
            ->post('https://www.google.com/recaptcha/api/siteverify', [
                'secret' => $secret,
                'response' => $response,
                'remoteip' => $this->ip(),
            ]);

        $data = $verification->json();

        if (! $verification->ok() || ! ($data['success'] ?? false)) {
            SecurityAuditLogger::log('captcha_failed', 'failed', $this->resolveUser(), $this, [
                'email' => $this->string('email')->toString(),
                'errors' => $data['error-codes'] ?? [],
            ]);

            throw ValidationException::withMessages([
                'g-recaptcha-response' => 'Captcha verification failed.',
            ]);
        }
    }

    private function maxAttempts(): int
    {
        $attempts = (int) SecuritySettings::getValue('max_login_attempts', 5);

        return max(1, $attempts);
    }

    private function decaySeconds(): int
    {
        $baseMinutes = (int) SecuritySettings::getValue('lockout_minutes', 1);
        $attempts = RateLimiter::attempts($this->throttleKey());

        // Exponential backoff: base * 2^(attempts-maxAttempts)
        // e.g. 1min, 2min, 4min, 8min, 16min capped at 60min
        $maxAttempts = $this->maxAttempts();
        $overAttempts = max(0, $attempts - $maxAttempts);
        $multiplier = min(pow(2, $overAttempts), 60);
        $minutes = max(1, $baseMinutes) * $multiplier;

        return (int) min($minutes, 60) * 60;
    }

    private function resolveUser(): ?User
    {
        $email = trim($this->string('email')->toString());

        if ($email === '') {
            return null;
        }

        return User::query()->where('email', $email)->first();
    }
}
