<?php

namespace App\Jobs;

use App\Models\Event;
use App\Services\FaceRecognitionService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;

class DeferredFaceIndex implements ShouldQueue, ShouldBeUnique
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    public $tries = 3;
    public $backoff = [30, 120];
    public $timeout = 600;
    public $uniqueFor = 120;

    public function __construct(
        public int $eventId
    ) {
    }

    public function uniqueId(): string
    {
        return 'face-index-' . $this->eventId;
    }

    public function handle(): void
    {
        if (! config('face.enabled', false)) {
            return;
        }

        $event = Event::find($this->eventId);
        if (! $event) {
            return;
        }

        app(FaceRecognitionService::class)->index($event, false);

        // Verify indexing actually produced results — if not, let the queue retry
        $imageCount = $event->media()->where('file_type', 'image')->where('status', 'ready')->count();
        $faceCount = DB::table('event_media_faces')->where('event_id', $event->id)->count();

        if ($imageCount > 0 && $faceCount === 0) {
            throw new \RuntimeException(
                "Face indexing produced 0 embeddings for {$imageCount} images (event {$this->eventId}). Will retry."
            );
        }
    }
}
