<?php

namespace App\Jobs;

use App\Models\EventMedia;
use App\Models\SystemSetting;
use App\Services\CloudStorageManager;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Storage;

class MigrateClientStorageMedia implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    public int $timeout = 1200;

    public function __construct(private ?int $lastId = null, private int $batchSize = 50)
    {
    }

    public function handle(CloudStorageManager $manager): void
    {
        if (! $manager->hasProviderConfig()) {
            $this->updateMigrationState([
                'status' => 'failed',
                'last_error' => 'Cloud storage is not configured.',
                'finished_at' => now()->toDateTimeString(),
            ]);
            return;
        }

        $manager->registerDisks();

        $targetPreview = CloudStorageManager::PREVIEW_DISK;
        $targetOriginal = CloudStorageManager::ORIGINAL_DISK;

        $query = EventMedia::query()->orderBy('id');
        if ($this->lastId) {
            $query->where('id', '>', $this->lastId);
        }

        $items = $query->limit($this->batchSize)->get();
        if ($items->isEmpty()) {
            $this->updateMigrationState([
                'status' => 'completed',
                'finished_at' => now()->toDateTimeString(),
            ]);
            return;
        }

        $processed = 0;
        $failed = 0;

        foreach ($items as $media) {
            $processed++;

            if ($media->disk === $targetPreview && $media->originalDisk() === $targetOriginal) {
                continue;
            }

            $migrated = $this->migrateMedia($media, $targetPreview, $targetOriginal);
            if (! $migrated) {
                $failed++;
            }
        }

        $this->incrementMigrationState($processed, $failed, $items->last()?->id);

        self::dispatch($items->last()?->id, $this->batchSize);
    }

    private function migrateMedia(EventMedia $media, string $targetPreview, string $targetOriginal): bool
    {
        $sourcePreview = $media->disk;
        $sourceOriginal = $media->originalDisk();

        $originalOk = $this->copyFile($sourceOriginal, $targetOriginal, $media->original_path);
        $optimizedOk = $this->copyFile($sourcePreview, $targetPreview, $media->optimized_path);
        $thumbOk = $this->copyFile($sourcePreview, $targetPreview, $media->thumbnail_path);

        if (! $originalOk || ! $optimizedOk || ! $thumbOk) {
            return false;
        }

        $meta = is_array($media->meta) ? $media->meta : [];
        $meta['original_disk'] = $targetOriginal;

        $media->update([
            'disk' => $targetPreview,
            'meta' => $meta,
        ]);

        return true;
    }

    private function copyFile(string $sourceDisk, string $targetDisk, ?string $path): bool
    {
        if (! $path) {
            return true;
        }

        if ($sourceDisk === $targetDisk) {
            return true;
        }

        $source = Storage::disk($sourceDisk);
        $target = Storage::disk($targetDisk);

        if (! $source->exists($path)) {
            return false;
        }

        $stream = $source->readStream($path);
        if (! $stream) {
            return false;
        }

        $target->put($path, $stream);

        if (is_resource($stream)) {
            fclose($stream);
        }

        if (! $target->exists($path)) {
            return false;
        }

        $source->delete($path);

        return true;
    }

    private function updateMigrationState(array $updates): void
    {
        $setting = SystemSetting::query()->firstOrCreate([]);
        $payload = $setting->payload ?? [];
        $cloud = is_array($payload['cloud_storage'] ?? null) ? $payload['cloud_storage'] : [];
        $migration = is_array($cloud['migration'] ?? null) ? $cloud['migration'] : [];

        $cloud['migration'] = array_merge($migration, $updates);
        $payload['cloud_storage'] = $cloud;
        $setting->payload = $payload;
        $setting->save();
    }

    private function incrementMigrationState(int $processed, int $failed, ?int $lastId): void
    {
        $setting = SystemSetting::query()->firstOrCreate([]);
        $payload = $setting->payload ?? [];
        $cloud = is_array($payload['cloud_storage'] ?? null) ? $payload['cloud_storage'] : [];
        $migration = is_array($cloud['migration'] ?? null) ? $cloud['migration'] : [];

        $migration['processed'] = (int) ($migration['processed'] ?? 0) + $processed;
        $migration['failed'] = (int) ($migration['failed'] ?? 0) + $failed;
        $migration['last_id'] = $lastId;
        $migration['status'] = 'running';

        $cloud['migration'] = $migration;
        $payload['cloud_storage'] = $cloud;
        $setting->payload = $payload;
        $setting->save();
    }
}
