<?php

namespace App\Jobs;

use App\Models\EventMedia;
use App\Models\Plan;
use App\Models\User;
use App\Services\CloudStorageManager;
use App\Support\EventMediaSettings;
use App\Support\MediaObfuscator;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Schema;

class ProcessEventMedia implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * FIX #9: Explicit job timeout and retry configuration
     */
    public $timeout = 600; // 10 minutes max
    public $tries = 3; // Retry up to 3 times
    public $backoff = [60, 600, 3600]; // Exponential backoff: 1min, 10min, 1hour

    private bool $allowWatermark = false;
    private bool $skipOptimized = false;
    private ?string $watermarkPath = null;
    private ?string $watermarkPosition = null;
    private bool $watermarkCleanup = false;
    private ?array $watermarkImageCache = null;
    private static array $planCache = [];

    public function __construct(public int $mediaId)
    {
    }

    public function handle(): void
    {
        app(CloudStorageManager::class)->registerDisks();

        $media = EventMedia::with('event.createdBy')->find($this->mediaId);
        if (! $media) {
            return;
        }

        $this->allowWatermark = $this->shouldApplyWatermark($media);

        $originalDiskName = $media->originalDisk();
        $previewDiskName = $media->disk;
        $originalDisk = Storage::disk($originalDiskName);
        $previewDisk = Storage::disk($previewDiskName);

        if (! $originalDisk->exists($media->original_path)) {
            $media->update(['status' => 'failed']);
            return;
        }

        $media->update(['status' => 'processing']);

        // Videos: just copy original as optimized (no resize)
        if ($media->file_type === 'video') {
            $optimizedPath = $media->optimized_path;

            if (! $optimizedPath || ! $previewDisk->exists($optimizedPath)) {
                $optimizedPath = $this->buildVariantPath($media->event_id, $media->original_path, 'optimized');
                $previewDisk->makeDirectory(dirname($optimizedPath));
                $this->copyBetweenDisks($originalDiskName, $media->original_path, $previewDiskName, $optimizedPath);
            }

            if (! $optimizedPath && $previewDiskName === $originalDiskName) {
                $optimizedPath = $media->original_path;
            }

            // Obfuscate video files on disk
            MediaObfuscator::obfuscateOnDisk($originalDiskName, $media->original_path);
            if ($optimizedPath && $optimizedPath !== $media->original_path) {
                MediaObfuscator::obfuscateOnDisk($previewDiskName, $optimizedPath);
            }

            $media->update([
                'optimized_path' => $optimizedPath,
                'thumbnail_path' => null,
                'status' => 'ready',
                'meta' => array_merge($media->meta ?? [], ['obfuscated' => true]),
            ]);

            return;
        }

        // Images: create optimized variant only (resized + watermark if enabled)
        // No thumbnails — saves disk, CPU, and I/O per image
        $this->prepareWatermark($media);

        $optimizedPath = $media->optimized_path ?: $this->buildVariantPath($media->event_id, $media->original_path, 'optimized');
        $previewDisk->makeDirectory(dirname($optimizedPath));

        $sourcePath = $this->resolveSourcePath($originalDiskName, $media->original_path);
        $optimizedTarget = $this->resolveTargetPath($previewDiskName, $optimizedPath);

        // Check disk space and memory before processing
        $this->checkDiskSpace($media);
        $this->validateProcessingRequirements($sourcePath['path'], $media);

        $processed = $this->processImage($sourcePath['path'], $optimizedTarget['path']);

        // Cleanup cached watermark resource
        $this->destroyWatermarkCache();

        if ($processed) {
            if ($this->skipOptimized) {
                // Source <= maxWidth and no watermark — just use original
                $optimizedPath = $media->original_path;
            } elseif ($optimizedTarget['requires_upload']) {
                // Cloud: obfuscate temp before uploading
                MediaObfuscator::obfuscate($optimizedTarget['path']);
                $this->uploadFromTemp($previewDiskName, $optimizedPath, $optimizedTarget['path']);
            } else {
                // Local: obfuscate optimized in place
                MediaObfuscator::obfuscate($optimizedTarget['path']);
            }
        } else {
            // GD failed — fall back to using original as optimized
            $optimizedPath = $media->original_path;
        }

        if ($sourcePath['cleanup']) {
            @unlink($sourcePath['path']);
        }
        if (!$this->skipOptimized && $optimizedTarget['cleanup']) {
            @unlink($optimizedTarget['path']);
        }
        if ($this->watermarkCleanup && $this->watermarkPath) {
            @unlink($this->watermarkPath);
        }

        // Obfuscate original file (makes it unreadable on disk)
        MediaObfuscator::obfuscateOnDisk($originalDiskName, $media->original_path);

        $media->update([
            'optimized_path' => $optimizedPath,
            'thumbnail_path' => null,
            'status' => 'ready',
            'meta' => array_merge($media->meta ?? [], ['obfuscated' => true]),
        ]);

        // Dispatch deferred face indexing (runs once per event, not per photo)
        if (config('face.enabled', false)) {
            DeferredFaceIndex::dispatch($media->event_id)
                ->delay(now()->addSeconds(30))
                ->onQueue('default');
        }

    }

    private function copyBetweenDisks(string $fromDiskName, string $fromPath, string $toDiskName, string $toPath): void
    {
        if ($fromDiskName === $toDiskName) {
            Storage::disk($toDiskName)->copy($fromPath, $toPath);
            return;
        }

        $stream = Storage::disk($fromDiskName)->readStream($fromPath);
        if (! $stream) {
            return;
        }

        Storage::disk($toDiskName)->put($toPath, $stream);

        if (is_resource($stream)) {
            fclose($stream);
        }
    }

    private function buildVariantPath(int $eventId, string $originalPath, string $variant): string
    {
        return app(CloudStorageManager::class)->buildVariantPath($eventId, $originalPath, $variant);
    }

    private function resolveSourcePath(string $diskName, string $path): array
    {
        if ($this->isLocalDisk($diskName)) {
            $localPath = Storage::disk($diskName)->path($path);

            // Handle obfuscated files (e.g. job retry after partial obfuscation)
            if (MediaObfuscator::isObfuscated($localPath)) {
                $temp = MediaObfuscator::deobfuscateToTemp($localPath);
                return ['path' => $temp, 'cleanup' => true];
            }

            return ['path' => $localPath, 'cleanup' => false];
        }

        // Cloud disk: download to temp, auto-stripping obfuscation header
        $temp = $this->tempFilePath('source_');
        $stream = Storage::disk($diskName)->readStream($path);
        if ($stream) {
            $output = fopen($temp, 'wb');

            // Check for obfuscation header
            $magic = fread($stream, 8);
            if ($magic === MediaObfuscator::MAGIC) {
                fread($stream, 24); // skip rest of header
            } else {
                fwrite($output, $magic); // not obfuscated, write back
            }

            stream_copy_to_stream($stream, $output);
            fclose($output);
            if (is_resource($stream)) {
                fclose($stream);
            }
        }

        return ['path' => $temp, 'cleanup' => true];
    }

    private function resolveTargetPath(string $diskName, string $path): array
    {
        if ($this->isLocalDisk($diskName)) {
            return [
                'path' => Storage::disk($diskName)->path($path),
                'requires_upload' => false,
                'cleanup' => false,
            ];
        }

        return [
            'path' => $this->tempFilePath('target_'),
            'requires_upload' => true,
            'cleanup' => true,
        ];
    }

    private function uploadFromTemp(string $diskName, string $path, string $tempPath): void
    {
        $stream = fopen($tempPath, 'rb');
        Storage::disk($diskName)->put($path, $stream);

        if (is_resource($stream)) {
            fclose($stream);
        }
    }

    private function isLocalDisk(string $diskName): bool
    {
        return config('filesystems.disks.' . $diskName . '.driver') === 'local';
    }

    private function tempFilePath(string $prefix): string
    {
        $dir = storage_path('app/tmp/media-processing');
        File::ensureDirectoryExists($dir);

        return tempnam($dir, $prefix);
    }

    private function processImage(string $source, string $optimizedPath): bool
    {
        if (! extension_loaded('gd')) {
            return false;
        }

        $info = @getimagesize($source);
        if (! $info) {
            return false;
        }

        $type = $info[2] ?? null;
        $sourceImage = $this->createImage($source, $type);
        if (! $sourceImage) {
            return false;
        }

        // Apply EXIF orientation in-memory (single decode, no re-save to disk)
        $sourceImage = $this->applyExifOrientation($source, $sourceImage);
        $width = imagesx($sourceImage);
        $height = imagesy($sourceImage);

        $maxWidth = (int) EventMediaSettings::getValue('max_width', config('events.media.max_width', 2000));

        // Skip if source already fits and no watermark needed
        $needsOptimized = $width > $maxWidth || $this->allowWatermark;
        if ($needsOptimized) {
            $this->resizeAndSave($sourceImage, $width, $height, $maxWidth, $optimizedPath, $type, true);
        }

        imagedestroy($sourceImage);
        $this->skipOptimized = !$needsOptimized;

        return true;
    }

    /**
     * Apply EXIF orientation rotation in-memory (replaces preserveOrientation disk rewrite)
     */
    private function applyExifOrientation(string $path, $image)
    {
        if (!function_exists('exif_read_data')) {
            return $image;
        }

        try {
            $exif = @exif_read_data($path);
            if (!$exif || !isset($exif['Orientation']) || (int) $exif['Orientation'] <= 1) {
                return $image;
            }

            $rotated = match ((int) $exif['Orientation']) {
                3 => imagerotate($image, 180, 0),
                6 => imagerotate($image, -90, 0),
                8 => imagerotate($image, 90, 0),
                default => null,
            };

            if ($rotated) {
                imagedestroy($image);
                return $rotated;
            }
        } catch (\Throwable $e) {
            // EXIF orientation failed — continue with original orientation
        }

        return $image;
    }

    private function createImage(string $source, ?int $type)
    {
        return match ($type) {
            IMAGETYPE_JPEG => @imagecreatefromjpeg($source),
            IMAGETYPE_PNG => @imagecreatefrompng($source),
            IMAGETYPE_WEBP => function_exists('imagecreatefromwebp') ? @imagecreatefromwebp($source) : null,
            default => null,
        };
    }

    private function resizeAndSave($sourceImage, int $width, int $height, int $targetWidth, string $path, int $type, bool $applyWatermark): void
    {
        $scale = $width > $targetWidth ? $targetWidth / $width : 1;
        $newWidth = (int) round($width * $scale);
        $newHeight = (int) round($height * $scale);

        $canvas = imagecreatetruecolor($newWidth, $newHeight);
        $this->preserveTransparency($canvas, $type);
        imagecopyresampled($canvas, $sourceImage, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);

        if ($applyWatermark && $this->allowWatermark) {
            $this->applyImageWatermark($canvas);
        }

        $this->saveImage($canvas, $path, $type);
        imagedestroy($canvas);
    }

    private function preserveTransparency($canvas, int $type): void
    {
        if (in_array($type, [IMAGETYPE_PNG, IMAGETYPE_WEBP], true)) {
            imagealphablending($canvas, false);
            imagesavealpha($canvas, true);
        }
    }

    private function applyImageWatermark($image): void
    {
        if (! $this->watermarkPath || ! is_file($this->watermarkPath)) {
            return;
        }

        $imgWidth = imagesx($image);
        $imgHeight = imagesy($image);
        if ($imgWidth <= 0 || $imgHeight <= 0) {
            return;
        }

        $watermarkData = $this->loadWatermarkImage();
        if (! $watermarkData) {
            return;
        }

        $watermark = $watermarkData['image'];
        $wmWidth = $watermarkData['width'];
        $wmHeight = $watermarkData['height'];

        $maxWidth = (int) round($imgWidth * 0.22);
        $maxHeight = (int) round($imgHeight * 0.22);

        [$watermark, $wmWidth, $wmHeight, $scaled] = $this->scaleWatermark($watermark, $wmWidth, $wmHeight, $maxWidth, $maxHeight);

        $margin = max(12, (int) round(min($imgWidth, $imgHeight) * 0.03));
        [$x, $y] = $this->resolveWatermarkPosition($imgWidth, $imgHeight, $wmWidth, $wmHeight, $margin);

        imagealphablending($image, true);
        imagesavealpha($image, true);
        imagealphablending($watermark, true);
        imagesavealpha($watermark, true);

        imagecopy($image, $watermark, $x, $y, 0, 0, $wmWidth, $wmHeight);

        // Only destroy the scaled copy, not the cached original
        if ($scaled) {
            imagedestroy($watermark);
        }
    }

    private function prepareWatermark(EventMedia $media): void
    {
        if (! $this->allowWatermark) {
            return;
        }

        $watermarkImage = trim((string) EventMediaSettings::getValue('watermark.image', config('events.media.watermark.image', '')));
        if ($watermarkImage === '') {
            $this->allowWatermark = false;
            return;
        }

        $positions = config('events.media.watermark.positions', []);
        $eventPosition = is_string($media->event?->watermark_position) ? trim($media->event->watermark_position) : '';
        $position = $eventPosition !== ''
            ? $eventPosition
            : (string) EventMediaSettings::getValue('watermark.position', config('events.media.watermark.position', 'top_right'));

        if ($positions && ! array_key_exists($position, $positions)) {
            $position = array_key_exists('top_right', $positions) ? 'top_right' : array_key_first($positions);
        }

        $resolved = $this->resolveWatermarkPath($watermarkImage);
        if (! $resolved['path']) {
            $this->allowWatermark = false;
            return;
        }

        $this->watermarkPath = $resolved['path'];
        $this->watermarkCleanup = $resolved['cleanup'];
        $this->watermarkPosition = $position ?: 'top_right';
    }

    private function resolveWatermarkPath(string $path): array
    {
        $disk = Storage::disk('public');
        if (! $disk->exists($path)) {
            return ['path' => null, 'cleanup' => false];
        }

        if (config('filesystems.disks.public.driver') === 'local') {
            return ['path' => $disk->path($path), 'cleanup' => false];
        }

        $temp = $this->tempFilePath('watermark_');
        $stream = $disk->readStream($path);
        if (! $stream) {
            return ['path' => null, 'cleanup' => false];
        }

        $output = fopen($temp, 'wb');
        stream_copy_to_stream($stream, $output);
        fclose($output);
        if (is_resource($stream)) {
            fclose($stream);
        }

        return ['path' => $temp, 'cleanup' => true];
    }

    private function loadWatermarkImage(): ?array
    {
        // Return cached watermark if available (avoid re-decoding per variant)
        if ($this->watermarkImageCache !== null) {
            return $this->watermarkImageCache;
        }

        if (! $this->watermarkPath || ! is_file($this->watermarkPath)) {
            return null;
        }

        $info = @getimagesize($this->watermarkPath);
        if (! $info) {
            return null;
        }

        $type = $info[2] ?? null;
        $image = $this->createImage($this->watermarkPath, $type);
        if (! $image) {
            return null;
        }

        $this->watermarkImageCache = [
            'image' => $image,
            'width' => (int) ($info[0] ?? 0),
            'height' => (int) ($info[1] ?? 0),
        ];

        return $this->watermarkImageCache;
    }

    private function destroyWatermarkCache(): void
    {
        if ($this->watermarkImageCache && isset($this->watermarkImageCache['image'])) {
            @imagedestroy($this->watermarkImageCache['image']);
            $this->watermarkImageCache = null;
        }
    }

    private function scaleWatermark($watermark, int $width, int $height, int $maxWidth, int $maxHeight): array
    {
        if ($width <= 0 || $height <= 0 || $maxWidth <= 0 || $maxHeight <= 0) {
            return [$watermark, $width, $height, false];
        }

        $scale = min(1, $maxWidth / $width, $maxHeight / $height);
        if ($scale >= 1) {
            return [$watermark, $width, $height, false];
        }

        $newWidth = max(1, (int) round($width * $scale));
        $newHeight = max(1, (int) round($height * $scale));

        $scaled = imagecreatetruecolor($newWidth, $newHeight);
        imagealphablending($scaled, false);
        imagesavealpha($scaled, true);
        $transparent = imagecolorallocatealpha($scaled, 0, 0, 0, 127);
        imagefill($scaled, 0, 0, $transparent);
        imagecopyresampled($scaled, $watermark, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);

        return [$scaled, $newWidth, $newHeight, true];
    }

    private function resolveWatermarkPosition(int $imgWidth, int $imgHeight, int $wmWidth, int $wmHeight, int $margin): array
    {
        $position = $this->watermarkPosition ?? 'top_right';

        $x = match ($position) {
            'top_left', 'center_left', 'bottom_left' => $margin,
            'top_center', 'center', 'bottom_center' => (int) round(($imgWidth - $wmWidth) / 2),
            'top_right', 'center_right', 'bottom_right' => $imgWidth - $wmWidth - $margin,
            default => $imgWidth - $wmWidth - $margin,
        };

        $y = match ($position) {
            'top_left', 'top_center', 'top_right' => $margin,
            'center_left', 'center', 'center_right' => (int) round(($imgHeight - $wmHeight) / 2),
            'bottom_left', 'bottom_center', 'bottom_right' => $imgHeight - $wmHeight - $margin,
            default => $margin,
        };

        $x = max(0, min($x, $imgWidth - $wmWidth));
        $y = max(0, min($y, $imgHeight - $wmHeight));

        return [$x, $y];
    }

    private function saveImage($image, string $path, int $type): void
    {
        $quality = (int) EventMediaSettings::getValue('quality', config('events.media.quality', 82));

        match ($type) {
            IMAGETYPE_JPEG => imagejpeg($image, $path, $quality),
            IMAGETYPE_PNG => imagepng($image, $path, 6),
            IMAGETYPE_WEBP => function_exists('imagewebp') ? imagewebp($image, $path, $quality) : imagejpeg($image, $path, $quality),
            default => imagejpeg($image, $path, $quality),
        };
    }

    private function shouldApplyWatermark(EventMedia $media): bool
    {
        $globalWatermarkEnabled = filter_var(
            EventMediaSettings::getValue('watermark.enabled', config('events.media.watermark.enabled', true)),
            FILTER_VALIDATE_BOOLEAN
        );

        if (! $globalWatermarkEnabled) {
            return false;
        }

        // Check if a watermark image is actually configured
        $watermarkImage = trim((string) EventMediaSettings::getValue('watermark.image', config('events.media.watermark.image', '')));
        if ($watermarkImage === '') {
            return false;
        }

        // Apply based on Event Creator's role/plan settings
        $owner = $media->event->createdBy ?? null;
        if (! $owner) {
            return true; // Default to watermark if owner is missing
        }

        // Admin/Super Admin — always apply watermark when globally enabled
        if (method_exists($owner, 'hasAnyRole') && $owner->hasAnyRole(['Super Admin', 'Admin'])) {
            return true;
        }

        $plan = $this->resolvePlanFromUser($owner);
        if ($plan instanceof Plan) {
            return (bool) $plan->has_watermark;
        }

        // Falls back to free plan logic if plan record is missing
        return $this->isFreePlan($owner);
    }

    private function isFreePlan(?User $user): bool
    {
        if (! $user) {
            return true;
        }

        $plan = $this->resolvePlanFromUser($user);

        if ($plan instanceof Plan) {
            return $this->isFreePlanRecord($plan);
        }

        if (is_string($plan) && $plan !== '') {
            return $this->isFreeLabel($plan);
        }

        return $plan === null ? true : (bool) $plan;
    }

    private static ?bool $plansTableExists = null;

    private function plansTableExists(): bool
    {
        if (self::$plansTableExists === null) {
            self::$plansTableExists = Schema::hasTable('plans');
        }
        return self::$plansTableExists;
    }

    private function resolvePlanFromUser(User $user): Plan|string|bool|null
    {
        // Cache plan resolution per user across sequential jobs in same worker
        $cacheKey = $user->id;
        if (array_key_exists($cacheKey, self::$planCache)) {
            return self::$planCache[$cacheKey];
        }

        $result = $this->doResolvePlanFromUser($user);
        self::$planCache[$cacheKey] = $result;
        return $result;
    }

    private function doResolvePlanFromUser(User $user): Plan|string|bool|null
    {
        if (method_exists($user, 'plan')) {
            $relatedPlan = $user->plan;
            if ($relatedPlan instanceof Plan) {
                return $relatedPlan;
            }
        }

        if (!$this->plansTableExists()) {
            return null;
        }

        $planId = $user->getAttribute('plan_id');
        if ($planId) {
            $plan = Plan::query()->find($planId);
            if ($plan) {
                return $plan;
            }
        }

        $planSlug = $user->getAttribute('plan_slug') ?: $user->getAttribute('plan');
        if (is_string($planSlug) && $planSlug !== '') {
            $plan = Plan::query()->where('slug', $planSlug)->first();
            if ($plan) {
                return $plan;
            }
        }

        $planName = $user->getAttribute('plan_name');
        if (is_string($planName) && $planName !== '') {
            $plan = Plan::query()->where('name', $planName)->first();
            if ($plan) {
                return $plan;
            }
        }

        return null;
    }

    private function isFreePlanRecord(Plan $plan): bool
    {
        $price = $plan->price;
        if ($price !== null && (float) $price <= 0.0) {
            return true;
        }

        return $this->isFreeLabel($plan->slug) || $this->isFreeLabel($plan->name);
    }

    private function isFreeLabel(?string $value): bool
    {
        if ($value === null) {
            return false;
        }

        return strtolower(trim($value)) === 'free';
    }

    /**
     * FIX #4: Check disk space before processing to prevent corrupted files
     */
    private function checkDiskSpace(EventMedia $media): void
    {
        $previewDisk = Storage::disk($media->disk);
        $diskPath = $previewDisk->path('');

        // Estimate space needed:
        // - Optimized variant: ~60% of original (JPEG compression)
        // - Safety buffer: 100MB
        $estimatedSpace = ($media->size * 0.65) + (100 * 1024 * 1024);

        $freeSpace = @disk_free_space($diskPath);

        if ($freeSpace === false) {
            return;
        }

        if ($freeSpace < $estimatedSpace) {
            $freeMB = round($freeSpace / 1024 / 1024, 1);
            $requiredMB = round($estimatedSpace / 1024 / 1024, 1);

            throw new \RuntimeException(
                "Insufficient disk space. Available: {$freeMB}MB, Required: {$requiredMB}MB"
            );
        }
    }

    /**
     * FIX #3 & #6: Validate memory and GD extension before processing
     */
    private function validateProcessingRequirements(string $sourcePath, EventMedia $media): void
    {
        if (!extension_loaded('gd')) {
            throw new \RuntimeException('GD extension is required for image processing but is not loaded');
        }

        $dimensions = @getimagesize($sourcePath);
        if ($dimensions === false) {
            return; // Not an image
        }

        [$width, $height] = $dimensions;

        // Estimate memory needed: width * height * 4 bytes (RGBA) * 3 (source + optimized + overhead)
        $estimatedMemory = $width * $height * 4 * 3;

        // Get PHP memory limit
        $memoryLimit = ini_get('memory_limit');
        if ($memoryLimit === '-1') {
            return; // Unlimited memory
        }

        $memoryLimitBytes = $this->parseMemoryLimit($memoryLimit);
        $memoryUsed = memory_get_usage(true);
        $memoryAvailable = $memoryLimitBytes - $memoryUsed;

        if ($estimatedMemory > $memoryAvailable) {
            // Try to increase memory limit dynamically
            $needed = $memoryUsed + (int) ($estimatedMemory * 1.2);
            $neededMB = (int) ceil($needed / 1024 / 1024);
            $maxAllowed = 1024; // 1GB ceiling
            if ($neededMB <= $maxAllowed) {
                @ini_set('memory_limit', $neededMB . 'M');
                return;
            }

            $estimatedMB = round($estimatedMemory / 1024 / 1024, 1);
            $availableMB = round($memoryAvailable / 1024 / 1024, 1);

            throw new \RuntimeException(
                "Insufficient memory to process {$width}x{$height} image. " .
                "Needs {$estimatedMB}MB, only {$availableMB}MB available."
            );
        }
    }

    private function parseMemoryLimit(string $limit): int
    {
        $limit = trim($limit);
        $last = strtolower($limit[\strlen($limit) - 1]);
        $value = (int) $limit;

        switch ($last) {
            case 'g': $value *= 1024;
            case 'm': $value *= 1024;
            case 'k': $value *= 1024;
        }

        return $value;
    }

    /**
     * FIX #9: Handle job failure
     */
    public function failed(\Throwable $exception): void
    {
        $media = EventMedia::find($this->mediaId);
        if ($media) {
            $media->update([
                'status' => 'failed',
                'meta' => array_merge($media->meta ?? [], [
                    'error' => $exception->getMessage(),
                    'failed_at' => now()->toDateTimeString(),
                ]),
            ]);
        }
    }

}
