<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Models\DriveImport;

class Event extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'type',
        'details',
        'event_date',
        'expiry_date',
        'notifications_enabled',
        'guest_pin',
        'admin_pin',
        'share_token',
        'hashtags',
        'is_active',
        'allow_guest_upload',
        'allow_public_downloads',
        'created_by',
        'design_settings',
        'watermark_position',
        'storage_quota_bytes',
        'storage_used_bytes',
    ];

    protected $casts = [
        'event_date' => 'date',
        'expiry_date' => 'date',
        'notifications_enabled' => 'boolean',
        'is_active' => 'boolean',
        'allow_guest_upload' => 'boolean',
        'allow_public_downloads' => 'boolean',
        'design_settings' => 'array',
        'hashtags' => 'array',
        'storage_quota_bytes' => 'integer',
        'storage_used_bytes' => 'integer',
    ];

    public function media()
    {
        return $this->hasMany(EventMedia::class);
    }

    public function coverMedia()
    {
        return $this->hasOne(EventMedia::class)->where('is_cover', true);
    }

    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function driveImports()
    {
        return $this->hasMany(DriveImport::class);
    }

    public function getShareUrlAttribute(): string
    {
        return route('events.share', $this->slug) . '?token=' . $this->share_token;
    }

    /**
     * Cleanup face recognition data when event is deleted.
     */
    protected static function booted(): void
    {
        static::deleting(function (Event $event) {
            // Delete all face recognition data in a single query (avoids N+1)
            \App\Models\EventMediaFace::where('event_id', $event->id)->delete();
        });
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true)
                     ->where(function ($q) {
                         $q->whereNull('expiry_date')
                           ->orWhere('expiry_date', '>=', now());
                     });
    }

    public function scopeExpired($query)
    {
        return $query->whereNotNull('expiry_date')
                     ->where('expiry_date', '<', now());
    }

    public function scopeForUser($query, $userId)
    {
        return $query->where('created_by', $userId);
    }
}
