<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes; // FIX #16
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\URL;

class EventMedia extends Model
{
    use HasFactory, SoftDeletes; // FIX #16

    protected $fillable = [
        'event_id',
        'disk',
        'original_path',
        'optimized_path',
        'thumbnail_path',
        'file_name',
        'file_type',
        'mime_type',
        'size',
        'file_hash', // FIX #18: Duplicate detection
        'width',
        'height',
        'duration',
        'is_featured',
        'is_cover',
        'sort_order',
        'status',
        'is_guest_upload',
        'meta',
    ];

    protected $casts = [
        'is_featured' => 'boolean',
        'is_cover' => 'boolean',
        'is_guest_upload' => 'boolean',
        'size' => 'integer',
        'width' => 'integer',
        'height' => 'integer',
        'duration' => 'integer',
        'meta' => 'array',
    ];

    public function event()
    {
        return $this->belongsTo(Event::class);
    }

    public function faces()
    {
        return $this->hasMany(EventMediaFace::class);
    }

    public function getPreviewUrlAttribute(): string
    {
        $path = $this->optimized_path ?: $this->original_path;

        // When optimized_path == original_path, the file lives on the original disk
        if ($this->optimized_path && $this->optimized_path === $this->original_path) {
            return $this->buildStorageUrl($this->originalDisk(), $path, 'optimized');
        }

        // No optimized yet (pending/processing) — show original from its actual disk
        if (! $this->optimized_path) {
            return $this->buildStorageUrl($this->originalDisk(), $this->original_path, 'original');
        }

        return $this->buildStorageUrl($this->disk, $path, 'optimized');
    }

    public function getThumbnailUrlAttribute(): string
    {
        if (! $this->thumbnail_path) {
            return '';
        }

        return $this->buildStorageUrl($this->disk, $this->thumbnail_path, 'thumbnail');
    }

    /**
     * Generate a signed URL with custom expiry (in minutes).
     * Use this on the share page so images stay loadable during long browsing sessions.
     */
    public function signedPreviewUrl(int $expiryMinutes = 240): string
    {
        $path = $this->optimized_path ?: $this->original_path;

        if ($this->optimized_path && $this->optimized_path === $this->original_path) {
            return $this->buildStorageUrl($this->originalDisk(), $path, 'optimized', $expiryMinutes);
        }

        // No optimized yet (pending/processing) — show original from its actual disk
        if (! $this->optimized_path) {
            return $this->buildStorageUrl($this->originalDisk(), $this->original_path, 'original', $expiryMinutes);
        }

        return $this->buildStorageUrl($this->disk, $path, 'optimized', $expiryMinutes);
    }

    public function signedThumbnailUrl(int $expiryMinutes = 240): string
    {
        if (! $this->thumbnail_path) {
            return '';
        }

        return $this->buildStorageUrl($this->disk, $this->thumbnail_path, 'thumbnail', $expiryMinutes);
    }

    public function originalDisk(): string
    {
        $meta = is_array($this->meta) ? $this->meta : [];
        $disk = $meta['original_disk'] ?? null;

        return $disk ?: $this->disk;
    }

    protected static function booted(): void
    {
        // FIX #14: Update event storage usage when media is created
        static::created(function (EventMedia $media) {
            $event = $media->event;
            if ($event && $media->size) {
                $event->increment('storage_used_bytes', $media->size);
            }
        });

        // FIX #16: Only delete files on force delete (permanent deletion)
        // Soft delete keeps files for recovery
        static::forceDeleting(function (EventMedia $media) {
            // FIX #14: Decrement storage usage on permanent deletion
            $event = $media->event;
            if ($event && $media->size) {
                $event->decrement('storage_used_bytes', $media->size);
            }

            // Delete face recognition data
            $media->faces()->delete();

            // Delete physical files from ALL disks
            $originalDisk = $media->originalDisk();
            $previewDisk = $media->disk;

            // Collect all unique paths per disk to avoid deleting same file twice
            // (optimized_path may equal original_path when image was small)
            $originalPaths = [];
            $previewPaths = [];

            if ($media->original_path) {
                $originalPaths[] = $media->original_path;
            }

            if ($media->optimized_path && $media->optimized_path !== $media->original_path) {
                $previewPaths[] = $media->optimized_path;
            }
            if ($media->thumbnail_path) {
                $previewPaths[] = $media->thumbnail_path;
            }

            foreach ($originalPaths as $path) {
                try {
                    Storage::disk($originalDisk)->delete($path);
                } catch (\Throwable $e) {
                    \Log::warning('Failed to delete original file', [
                        'media_id' => $media->id, 'path' => $path, 'disk' => $originalDisk,
                    ]);
                }
            }

            foreach ($previewPaths as $path) {
                try {
                    Storage::disk($previewDisk)->delete($path);
                } catch (\Throwable $e) {
                    \Log::warning('Failed to delete preview file', [
                        'media_id' => $media->id, 'path' => $path, 'disk' => $previewDisk,
                    ]);
                }
            }
        });

        // FIX #16: When restoring soft-deleted media, restore storage usage
        static::restored(function (EventMedia $media) {
            $event = $media->event;
            if ($event && $media->size) {
                $event->increment('storage_used_bytes', $media->size);
            }
        });
    }

    public function scopeReady($query)
    {
        return $query->where('status', 'ready');
    }

    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeFailed($query)
    {
        return $query->where('status', 'failed');
    }

    public function scopeImages($query)
    {
        return $query->where('file_type', 'image');
    }

    public function scopeVideos($query)
    {
        return $query->where('file_type', 'video');
    }

    private function buildStorageUrl(string $disk, string $path, string $variant, int $expiryMinutes = 30): string
    {
        $isObfuscated = $this->meta['obfuscated'] ?? false;

        // CDN and presigned URLs serve raw bytes — skip for obfuscated files
        // (they need header stripping via MediaPreviewController)
        if (! $isObfuscated) {
            // If a CDN URL is configured (CloudFront, Bunny CDN, etc.), use it directly
            $cdnUrl = config("filesystems.disks.{$disk}.cdn_url");
            if ($cdnUrl) {
                return rtrim($cdnUrl, '/') . '/' . ltrim($path, '/');
            }

            // For cloud disks (S3, GCS, etc.), use native presigned URLs
            $adapter = Storage::disk($disk);
            if (method_exists($adapter, 'temporaryUrl')) {
                try {
                    return $adapter->temporaryUrl($path, now()->addMinutes($expiryMinutes));
                } catch (\Throwable $exception) {
                    // Fall through to signed route
                }
            }
        }

        // Signed route — MediaPreviewController handles deobfuscation
        return URL::temporarySignedRoute('media.preview', now()->addMinutes($expiryMinutes), [
            'media' => $this->id,
            'variant' => $variant,
        ]);
    }
}
