<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Crypt;

class PaymentGateway extends Model
{
    protected $fillable = [
        'slug',
        'name',
        'description',
        'is_active',
        'credentials',
        'mode',
        'extra_config',
        'supported_currencies',
        'sort_order',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'extra_config' => 'array',
        'supported_currencies' => 'array',
    ];

    /**
     * Fields that should never appear in serialized output.
     */
    protected $hidden = [
        'credentials',
    ];

    /**
     * Scope to only active gateways.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Get all orders processed through this gateway.
     */
    public function orders(): HasMany
    {
        return $this->hasMany(Order::class);
    }

    /**
     * Decrypt the stored credentials and return as an associative array.
     */
    public function getDecryptedCredentials(): array
    {
        if (empty($this->credentials)) {
            return [];
        }

        try {
            $decrypted = Crypt::decryptString($this->credentials);

            return json_decode($decrypted, true) ?: [];
        } catch (\Illuminate\Contracts\Encryption\DecryptException $e) {
            report($e);

            return [];
        }
    }

    /**
     * Encrypt an array of credentials and store on the model.
     */
    public function setEncryptedCredentials(array $data): void
    {
        $this->credentials = Crypt::encryptString(json_encode($data));
        $this->save();
    }

    /**
     * Retrieve a single credential value by key from the decrypted credentials.
     */
    public function getCredential(string $key, $default = null)
    {
        $credentials = $this->getDecryptedCredentials();

        return $credentials[$key] ?? $default;
    }

    /**
     * Check whether this gateway is running in live/production mode.
     */
    public function isLive(): bool
    {
        return $this->mode === 'live';
    }

    /**
     * Check whether this gateway is running in sandbox/test mode.
     */
    public function isSandbox(): bool
    {
        return $this->mode === 'sandbox';
    }
}
