<?php

namespace App\Observers;

use App\Models\EventMedia;
use App\Models\EventMediaLog;

/**
 * FIX #20: Observer to log all event media operations
 */
class EventMediaObserver
{
    /**
     * Handle the EventMedia "created" event.
     */
    public function created(EventMedia $eventMedia): void
    {
        EventMediaLog::log($eventMedia, 'upload', [
            'file_name' => $eventMedia->file_name,
            'file_type' => $eventMedia->file_type,
            'size' => $eventMedia->size,
        ]);
    }

    /**
     * Handle the EventMedia "updated" event.
     */
    public function updated(EventMedia $eventMedia): void
    {
        // Only log significant updates, not processing status changes
        if ($eventMedia->isDirty(['is_featured', 'is_cover', 'sort_order', 'file_name'])) {
            EventMediaLog::log($eventMedia, 'update', [
                'changed_fields' => array_keys($eventMedia->getDirty()),
            ]);
        }
    }

    /**
     * Handle the EventMedia "deleted" event (soft delete).
     */
    public function deleted(EventMedia $eventMedia): void
    {
        // Skip logging during forceDelete — the row is already gone from DB
        if ($eventMedia->isForceDeleting()) {
            return;
        }

        EventMediaLog::log($eventMedia, 'delete', [
            'file_name' => $eventMedia->file_name,
            'soft_delete' => true,
        ]);
    }

    /**
     * Handle the EventMedia "forceDeleted" event (permanent deletion).
     */
    public function forceDeleted(EventMedia $eventMedia): void
    {
        // Row is already deleted from DB at this point, so FK insert would fail.
        // The cascade on the event_media_logs table handles cleanup automatically.
    }

    /**
     * Handle the EventMedia "restored" event.
     */
    public function restored(EventMedia $eventMedia): void
    {
        EventMediaLog::log($eventMedia, 'restore', [
            'file_name' => $eventMedia->file_name,
        ]);
    }
}
