<?php

namespace App\Providers;

use App\Models\EventMedia; // FIX #20
use App\Models\Language;
use App\Models\SystemSetting;
use App\Observers\EventMediaObserver; // FIX #20
use App\Services\CloudStorageManager;
use App\Support\LocaleSettings;
use Google\Client as GoogleClient;
use Google\Cloud\Storage\StorageClient;
use Google\Service\Drive as GoogleDriveService;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\View;
use Illuminate\Support\ServiceProvider;
use Illuminate\Validation\Rules\Password;
use League\Flysystem\AzureBlobStorage\AzureBlobStorageAdapter;
use League\Flysystem\Filesystem;
use League\Flysystem\GoogleCloudStorage\GoogleCloudStorageAdapter;
use Masbug\Flysystem\GoogleDriveAdapter;
use MicrosoftAzure\Storage\Blob\BlobRestProxy;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
            // R5: Bind CloudStorageManager as a singleton
            $this->app->singleton(CloudStorageManager::class, function ($app) {
                return new CloudStorageManager();
            });
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        // R5: Consistent password complexity across all forms
        Password::defaults(function () {
            return Password::min(8)->mixedCase()->numbers()->symbols();
        });

        Gate::policy(\App\Models\NewsletterSubscriber::class, \App\Policies\NewsletterSubscriberPolicy::class);

        $payload = $this->resolveSystemSettingsPayload();
        $generalSettings = $this->resolveGeneralSettings($payload);
        $gdprSettings = $this->resolveGdprSettings($payload);
        $extensionSettings = $this->resolveExtensionSettings($payload);
        $siteSettings = $this->resolveSiteSettings($payload);
        $seoSettings = $this->resolveSeoSettings($payload);
        $footerSettings = $this->resolveFooterSettings($payload);
        $smtpSettings = $this->resolveSmtpSettings($payload);
        $pwaSettings = $this->resolvePwaSettings($payload);

        $gdprConsent = null;
        if (app()->bound('request')) {
            $gdprConsent = request()->cookie('gdpr_consent');
        }

        $this->registerFilesystemDrivers();
        $this->configureMail($smtpSettings);
        $this->configurePublicDiskUrl();
        $this->configureCloudStorageDisks();
        $this->applyTimezone($generalSettings);
        $this->configurePwa($pwaSettings);

        // FIX #20: Register audit logging observer
        EventMedia::observe(EventMediaObserver::class);

        $this->ensureStorageLink();

        View::share([
            'generalSettings' => $generalSettings,
            'gdprSettings' => $gdprSettings,
            'extensionSettings' => $extensionSettings,
            'siteSettings' => $siteSettings,
            'seoSettings' => $seoSettings,
            'footerSettings' => $footerSettings,
            'gdprConsent' => $gdprConsent,
            'pwaSettings' => $pwaSettings,
        ]);

        View::composer(['layouts.admin', 'layouts.frontend'], function ($view): void {
            $multiLanguageEnabled = LocaleSettings::isEnabled();
            $languageOptions = collect();
            $currentLocale = app()->getLocale();
            $languageDirection = 'ltr';

            if ($this->dbHasTable('languages')) {
                $languages = Language::query()
                    ->where('is_active', true)
                    ->orderBy('name')
                    ->get(['name', 'code', 'native_name', 'direction']);

                $matched = $languages->firstWhere('code', $currentLocale);
                if ($matched && $matched->direction) {
                    $languageDirection = $matched->direction;
                }

                if ($multiLanguageEnabled) {
                    $languageOptions = $languages;
                }
            }

            $view->with([
                'languageOptions' => $languageOptions,
                'currentLocale' => $currentLocale,
                'multiLanguageEnabled' => $multiLanguageEnabled,
                'languageDirection' => $languageDirection,
            ]);
        });
    }

    private function registerFilesystemDrivers(): void
    {
        if (class_exists(StorageClient::class) && class_exists(GoogleCloudStorageAdapter::class)) {
            Storage::extend('gcs', function ($app, $config) {
                $storage = new StorageClient([
                    'projectId' => $config['project_id'] ?? null,
                    'keyFile' => $config['key_file'] ?? null,
                ]);
                $bucket = $storage->bucket($config['bucket'] ?? '');
                $adapter = new GoogleCloudStorageAdapter($bucket, $config['root'] ?? null);
                $filesystem = new Filesystem($adapter);

                return new \Illuminate\Filesystem\FilesystemAdapter($filesystem, $adapter, $config);
            });
        }

        if (class_exists(GoogleClient::class) && class_exists(GoogleDriveAdapter::class)) {
            Storage::extend('google', function ($app, $config) {
                $options = [];

                if (! empty($config['teamDriveId'] ?? null)) {
                    $options['teamDriveId'] = $config['teamDriveId'];
                }

                if (! empty($config['sharedFolderId'] ?? null)) {
                    $options['sharedFolderId'] = $config['sharedFolderId'];
                }

                $client = new GoogleClient();
                $client->setClientId($config['clientId'] ?? null);
                $client->setClientSecret($config['clientSecret'] ?? null);
                $client->refreshToken($config['refreshToken'] ?? null);
                $client->setApplicationName($config['applicationName'] ?? config('app.name', 'SnapNest'));

                $service = new GoogleDriveService($client);
                $adapter = new GoogleDriveAdapter($service, $config['folder'] ?? '/', $options);
                $filesystem = new Filesystem($adapter);

                return new \Illuminate\Filesystem\FilesystemAdapter($filesystem, $adapter, $config);
            });
        }

        if (class_exists(BlobRestProxy::class) && class_exists(AzureBlobStorageAdapter::class)) {
            Storage::extend('azure', function ($app, $config) {
                $accountName = $config['name'] ?? '';
                $accountKey = $config['key'] ?? '';
                $endpoint = trim((string) ($config['endpoint'] ?? ''));

                $connectionString = 'DefaultEndpointsProtocol=https;AccountName=' . $accountName . ';AccountKey=' . $accountKey . ';';
                $connectionString .= $endpoint !== ''
                    ? 'BlobEndpoint=' . $endpoint
                    : 'EndpointSuffix=core.windows.net';

                $client = BlobRestProxy::createBlobService($connectionString);
                $adapter = new AzureBlobStorageAdapter($client, $config['container'] ?? '', $config['root'] ?? null);
                $filesystem = new Filesystem($adapter);

                return new \Illuminate\Filesystem\FilesystemAdapter($filesystem, $adapter, $config);
            });
        }
    }

    private function configureCloudStorageDisks(): void
    {
        if (! $this->dbHasTable('system_settings')) {
            return;
        }

        app(CloudStorageManager::class)->registerDisks();
    }

    private function dbHasTable(string $table): bool
    {
        try {
            return Schema::hasTable($table);
        } catch (\Throwable) {
            return false;
        }
    }

    private function resolveSystemSettingsPayload(): array
    {
        if (! $this->dbHasTable('system_settings')) {
            return [];
        }

        $payload = SystemSetting::query()->value('payload');

        if (is_string($payload)) {
            $decoded = json_decode($payload, true);
            $payload = is_array($decoded) ? $decoded : [];
        }

        return is_array($payload) ? $payload : [];
    }

    private function resolveGdprSettings(array $payload): array
    {
        $defaults = [
            'gdpr_enabled' => false,
            'gdpr_message' => '',
            'gdpr_accept_label' => 'Accept',
            'gdpr_decline_label' => 'Decline',
        ];

        $stored = is_array($payload['gdpr'] ?? null) ? $payload['gdpr'] : [];

        return array_merge($defaults, $stored);
    }

    private function resolveGeneralSettings(array $payload): array
    {
        $defaults = [
            'default_language' => config('app.locale'),
            'default_currency' => '',
            'default_timezone' => config('app.timezone'),
            'date_format' => '',
            'time_format' => '',
            'support_email' => '',
            'support_phone' => '',
            'enable_multi_language' => false,
        ];

        $stored = is_array($payload['general'] ?? null) ? $payload['general'] : [];

        return array_merge($defaults, $stored);
    }

    private function resolveExtensionSettings(array $payload): array
    {
        $defaults = [
            'analytics' => [
                'google_enabled' => false,
                'google_measurement_id' => '',
            ],
            'adsense' => [
                'adsense_enabled' => false,
                'adsense_client_id' => '',
            ],
            'facebook' => [
                'pixel_enabled' => false,
                'pixel_id' => '',
            ],
            'microsoft' => [
                'uet_enabled' => false,
                'uet_tag_id' => '',
            ],
            'social' => [
                'og_enabled' => false,
                'og_title' => '',
                'og_description' => '',
                'og_image' => null,
            ],
        ];

        $stored = is_array($payload['extensions'] ?? null) ? $payload['extensions'] : [];

        return array_replace_recursive($defaults, $stored);
    }

    private function resolveSiteSettings(array $payload): array
    {
        $defaults = [
            'site_name' => '',
            'site_logo' => null,
            'admin_logo' => null,
            'favicon' => null,
            'right_click_disable' => false,
            'uc_browser_block' => false,
        ];

        $stored = is_array($payload['site'] ?? null) ? $payload['site'] : [];

        return array_merge($defaults, $stored);
    }

    private function resolveFooterSettings(array $payload): array
    {
        $defaults = [
            'footer_logo' => null,
            'footer_social' => [
                'facebook' => '',
                'x' => '',
                'linkedin' => '',
                'instagram' => '',
                'whatsapp' => '',
            ],
        ];

        $stored = is_array($payload['footer'] ?? null) ? $payload['footer'] : [];

        $result = array_merge($defaults, $stored);
        $result['footer_social'] = array_merge($defaults['footer_social'], is_array($result['footer_social'] ?? null) ? $result['footer_social'] : []);

        return $result;
    }

    private function resolveSeoSettings(array $payload): array
    {
        $defaults = [
            'meta_title' => '',
            'meta_description' => '',
            'meta_keywords' => '',
        ];

        $stored = is_array($payload['seo'] ?? null) ? $payload['seo'] : [];

        return array_merge($defaults, $stored);
    }

    private function resolveSmtpSettings(array $payload): array
    {
        $defaults = [
            'smtp_host' => '',
            'smtp_port' => '',
            'smtp_username' => '',
            'smtp_password' => '',
            'smtp_encryption' => '',
            'smtp_from_address' => '',
            'smtp_from_name' => '',
        ];

        $stored = is_array($payload['smtp'] ?? null) ? $payload['smtp'] : [];

        return array_merge($defaults, $stored);
    }

    private function configureMail(array $smtpSettings): void
    {
        $host = trim((string) ($smtpSettings['smtp_host'] ?? ''));
        $port = (int) ($smtpSettings['smtp_port'] ?? 0);
        $username = trim((string) ($smtpSettings['smtp_username'] ?? ''));
        $password = (string) ($smtpSettings['smtp_password'] ?? '');
        $encryption = trim((string) ($smtpSettings['smtp_encryption'] ?? ''));
        $fromAddress = trim((string) ($smtpSettings['smtp_from_address'] ?? ''));
        $fromName = trim((string) ($smtpSettings['smtp_from_name'] ?? ''));

        if ($host === '' && $port === 0 && $username === '' && $password === '' && $fromAddress === '' && $fromName === '') {
            return;
        }

        if ($host !== '' || $port > 0 || $username !== '' || $password !== '') {
            config(['mail.default' => 'smtp']);
        }

        if ($host !== '') {
            config(['mail.mailers.smtp.host' => $host]);
        }

        if ($port > 0) {
            config(['mail.mailers.smtp.port' => $port]);
        }

        if ($username !== '') {
            config(['mail.mailers.smtp.username' => $username]);
        }

        if ($password !== '') {
            config(['mail.mailers.smtp.password' => $password]);
        }

        if ($encryption !== '') {
            $scheme = $encryption === 'none' ? null : $encryption;
            config(['mail.mailers.smtp.scheme' => $scheme]);
        }

        if ($fromAddress !== '') {
            config(['mail.from.address' => $fromAddress]);
        }

        if ($fromName !== '') {
            config(['mail.from.name' => $fromName]);
        }
    }

    private function configurePublicDiskUrl(): void
    {
        if (! app()->bound('request')) {
            return;
        }

        $request = request();
        $baseUrl = rtrim($request->getSchemeAndHttpHost() . $request->getBasePath(), '/');
        $storagePath = public_path('storage');

        // Sync Shared Hosting Fix
        if (!is_link($storagePath)) {
            if (!file_exists($storagePath)) {
                @mkdir($storagePath, 0755, true);
            }
            config(['filesystems.disks.public.root' => $storagePath]);
        }

        config(['filesystems.disks.public.url' => $baseUrl . '/storage']);
    }

    private function applyTimezone(array $generalSettings): void
    {
        $timezone = trim((string) ($generalSettings['default_timezone'] ?? ''));

        if ($timezone === '') {
            return;
        }

        if (! in_array($timezone, timezone_identifiers_list(), true)) {
            return;
        }

        config(['app.timezone' => $timezone]);
        date_default_timezone_set($timezone);
    }

    private function resolvePwaSettings(array $payload): array
    {
        $defaults = [
            'enabled'          => false,
            'app_name'         => '',
            'app_url'          => '/',
            'icon'             => null,
            'splash_screen'    => null,
            'theme_color'      => '#156055',
            'background_color' => '#f5f0e8',
        ];

        $stored = is_array($payload['pwa'] ?? null) ? $payload['pwa'] : [];

        return array_merge($defaults, $stored);
    }

    private function configurePwa(array $pwaSettings): void
    {
        if (empty($pwaSettings['enabled'])) {
            return;
        }

        $appName = trim((string) ($pwaSettings['app_name'] ?? '')) ?: config('app.name', 'SnapNest');
        $startUrl = trim((string) ($pwaSettings['app_url'] ?? '')) ?: '/';
        $theme    = trim((string) ($pwaSettings['theme_color'] ?? '')) ?: '#156055';
        $bgColor  = trim((string) ($pwaSettings['background_color'] ?? '')) ?: '#f5f0e8';

        // Build icons in the format the package expects: ['72x72' => ['path' => '...', 'purpose' => '...']]
        if (! empty($pwaSettings['icon'])) {
            $iconUrl = asset('storage/' . $pwaSettings['icon']);
            $icons = [];
            foreach ([72, 96, 128, 144, 152, 192, 384, 512] as $size) {
                $icons["{$size}x{$size}"] = ['path' => $iconUrl, 'purpose' => 'any'];
            }
        } else {
            $icons = config('laravelpwa.manifest.icons', []);
        }

        // Splash: map user-uploaded splash to all required iOS sizes
        if (! empty($pwaSettings['splash_screen'])) {
            $splashUrl = asset('storage/' . $pwaSettings['splash_screen']);
            $splash = array_fill_keys([
                '640x1136', '750x1334', '828x1792', '1125x2436',
                '1242x2208', '1242x2688', '1536x2048',
                '1668x2224', '1668x2388', '2048x2732',
            ], $splashUrl);
        } else {
            $splash = config('laravelpwa.manifest.splash', []);
        }

        config([
            'laravelpwa.manifest.name'             => $appName,
            'laravelpwa.manifest.short_name'       => $appName,
            'laravelpwa.manifest.start_url'        => $startUrl,
            'laravelpwa.manifest.theme_color'      => $theme,
            'laravelpwa.manifest.background_color' => $bgColor,
            'laravelpwa.manifest.icons'            => $icons,
            'laravelpwa.manifest.splash'           => $splash,
            'laravelpwa.manifest.shortcuts'        => [],
        ]);
    }

    private function ensureStorageLink(): void
    {
        $storagePath = public_path('storage');

        // Check is extremely fast, only runs Artisan command if link is missing
        if (! is_link($storagePath) && ! file_exists($storagePath)) {
            try {
                \Illuminate\Support\Facades\Artisan::call('storage:link');
            } catch (\Throwable $e) {
                // Fail silently (likely file permission issue on some hostings)
            }
        }
    }

}
