<?php

namespace App\Services;

use App\Support\CloudStorageSettings;
use App\Support\EventMediaSettings;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class CloudStorageManager
{
    public const PREVIEW_DISK = 'client_media';
    public const ORIGINAL_DISK = 'client_media_original';

    private static bool $registered = false;

    public function registerDisks(): void
    {
        if (self::$registered) {
            return;
        }

        if (! $this->hasProviderConfig()) {
            return;
        }

        $config = $this->buildDiskConfig();
        if (! $config) {
            return;
        }

        config(['filesystems.disks.' . self::PREVIEW_DISK => $config]);
        config(['filesystems.disks.' . self::ORIGINAL_DISK => $config]);
        self::$registered = true;
    }

    public function shouldUseClientStorage(): bool
    {
        $enabled = (bool) CloudStorageSettings::getValue('enabled', false);

        return $enabled && $this->hasProviderConfig();
    }

    public function previewDiskName(): string
    {
        return $this->shouldUseClientStorage()
            ? self::PREVIEW_DISK
            : $this->defaultPreviewDisk();
    }

    public function originalDiskName(): string
    {
        return $this->shouldUseClientStorage()
            ? self::ORIGINAL_DISK
            : $this->defaultOriginalDisk();
    }

    public function hasProviderConfig(): bool
    {
        $provider = (string) CloudStorageSettings::getValue('provider', '');
        $providers = (array) CloudStorageSettings::getValue('providers', []);
        $config = (array) ($providers[$provider] ?? []);

        $config = $this->decryptProviderConfig($provider, $config);

        return match ($provider) {
            's3' => $this->hasAll($config, ['key', 'secret', 'region', 'bucket']),
            'gcs' => $this->hasAll($config, ['project_id', 'bucket', 'key_file']),
            'gdrive' => $this->hasAll($config, ['client_id', 'client_secret', 'refresh_token']),
            'azure' => $this->hasAll($config, ['account_name', 'account_key', 'container']),
            'bunny' => $this->hasAll($config, ['storage_zone', 'access_key']),
            'wasabi' => $this->hasAll($config, ['key', 'secret', 'region', 'bucket']),
            default => false,
        };
    }

    public function providerLabel(): string
    {
        return match ((string) CloudStorageSettings::getValue('provider', '')) {
            's3' => 'AWS S3',
            'gcs' => 'Google Cloud Storage',
            'gdrive' => 'Google Drive',
            'azure' => 'Azure Blob Storage',
            'bunny' => 'Bunny.net Storage',
            'wasabi' => 'Wasabi Storage',
            default => 'Unknown',
        };
    }

    public function pathPrefix(): string
    {
        $prefix = trim((string) CloudStorageSettings::getValue('base_path', ''));

        return trim($prefix, '/');
    }

    public function applyPrefix(string $path): string
    {
        $prefix = $this->pathPrefix();
        $path = ltrim($path, '/');

        if ($prefix === '') {
            return $path;
        }

        if (str_starts_with($path, $prefix . '/')) {
            return $path;
        }

        return $prefix . '/' . $path;
    }

    public function extractPrefixFromPath(string $path, int $eventId): string
    {
        $marker = 'events/' . $eventId . '/';
        $pos = strpos($path, $marker);

        if ($pos === false) {
            return '';
        }

        $prefix = trim(substr($path, 0, $pos), '/');

        return $prefix;
    }

    public function buildVariantPath(int $eventId, string $originalPath, string $variant): string
    {
        $prefix = $this->extractPrefixFromPath($originalPath, $eventId);
        $name = pathinfo($originalPath, PATHINFO_BASENAME);
        $path = 'events/' . $eventId . '/' . $variant . '/' . $name;

        return $prefix !== '' ? $prefix . '/' . $path : $path;
    }

    public function testConnection(): array
    {
        if (! $this->hasProviderConfig()) {
            return [
                'ok' => false,
                'message' => 'Storage configuration is incomplete. Please fill in the required fields.',
            ];
        }

        $this->registerDisks();

        try {
            $disk = Storage::disk(self::PREVIEW_DISK);
            $testPath = $this->applyPrefix('connection-tests/' . Str::random(12) . '.txt');
            $disk->put($testPath, 'Cloud storage connection test.');
            $disk->delete($testPath);
        } catch (\Throwable $exception) {
            return [
                'ok' => false,
                'message' => 'Connection failed. Please verify credentials, bucket access, and permissions.',
            ];
        }

        return [
            'ok' => true,
            'message' => 'Connection successful. Storage is reachable and writable.',
        ];
    }

    private function defaultPreviewDisk(): string
    {
        return (string) EventMediaSettings::getValue('disk', config('events.media.disk', 'public'));
    }

    private function defaultOriginalDisk(): string
    {
        return (string) EventMediaSettings::getValue('original_disk', config('events.media.original_disk', 'local'));
    }

    private function buildDiskConfig(): ?array
    {
        $provider = (string) CloudStorageSettings::getValue('provider', '');
        $providers = (array) CloudStorageSettings::getValue('providers', []);
        $config = (array) ($providers[$provider] ?? []);

        $config = $this->decryptProviderConfig($provider, $config);
        $visibility = (string) CloudStorageSettings::getValue('visibility', 'private');

        $baseConfig = [
            'visibility' => $visibility === 'public' ? 'public' : 'private',
            'throw' => false,
            'report' => false,
        ];

        $wasabiEndpoint = null;
        if ($provider === 'wasabi') {
            $wasabiEndpoint = $config['endpoint'] ?? null;
            $region = trim((string) ($config['region'] ?? ''));
            if ((! $wasabiEndpoint || trim((string) $wasabiEndpoint) === '') && $region !== '') {
                $wasabiEndpoint = 'https://s3.' . $region . '.wasabisys.com';
            }
        }

        return match ($provider) {
            's3' => array_merge($baseConfig, [
                'driver' => 's3',
                'key' => $config['key'] ?? '',
                'secret' => $config['secret'] ?? '',
                'region' => $config['region'] ?? '',
                'bucket' => $config['bucket'] ?? '',
                'endpoint' => $config['endpoint'] ?? null,
                'url' => $config['url'] ?? null,
                'cdn_url' => $config['cdn_url'] ?? null,
                'use_path_style_endpoint' => filter_var($config['use_path_style_endpoint'] ?? false, FILTER_VALIDATE_BOOLEAN),
            ]),
            'gcs' => array_merge($baseConfig, [
                'driver' => 'gcs',
                'project_id' => $config['project_id'] ?? '',
                'key_file' => $config['key_file'] ?? [],
                'bucket' => $config['bucket'] ?? '',
                'cdn_url' => $config['cdn_url'] ?? null,
            ]),
            'gdrive' => array_merge($baseConfig, [
                'driver' => 'google',
                'clientId' => $config['client_id'] ?? '',
                'clientSecret' => $config['client_secret'] ?? '',
                'refreshToken' => $config['refresh_token'] ?? '',
                'folder' => ! empty($config['folder']) ? $config['folder'] : '/',
                'teamDriveId' => $config['team_drive_id'] ?? null,
                'sharedFolderId' => $config['shared_folder_id'] ?? null,
                'applicationName' => $config['application_name'] ?? config('app.name', 'SnapNest'),
            ]),
            'azure' => array_merge($baseConfig, [
                'driver' => 'azure',
                'name' => $config['account_name'] ?? '',
                'key' => $config['account_key'] ?? '',
                'container' => $config['container'] ?? '',
                'endpoint' => $config['endpoint'] ?? null,
                'url' => $config['url'] ?? null,
                'cdn_url' => $config['cdn_url'] ?? null,
            ]),
            // Bunny uses S3-compatible API — key and secret are both the access_key
            // (Bunny doesn't distinguish between access key ID and secret)
            'bunny' => array_merge($baseConfig, [
                'driver' => 's3',
                'key' => $config['access_key'] ?? '',
                'secret' => $config['access_key'] ?? '',
                'region' => $config['region'] ?? 'auto',
                'bucket' => $config['storage_zone'] ?? '',
                'endpoint' => $config['endpoint'] ?? null,
                'url' => $config['url'] ?? null,
                'cdn_url' => $config['cdn_url'] ?? null,
                'use_path_style_endpoint' => true,
            ]),
            'wasabi' => array_merge($baseConfig, [
                'driver' => 's3',
                'key' => $config['key'] ?? '',
                'secret' => $config['secret'] ?? '',
                'region' => $config['region'] ?? '',
                'bucket' => $config['bucket'] ?? '',
                'endpoint' => $wasabiEndpoint,
                'url' => $config['url'] ?? null,
                'cdn_url' => $config['cdn_url'] ?? null,
                'use_path_style_endpoint' => filter_var($config['use_path_style_endpoint'] ?? false, FILTER_VALIDATE_BOOLEAN),
            ]),
            default => null,
        };
    }

    private function decryptProviderConfig(string $provider, array $config): array
    {
        $decrypt = fn ($value) => CloudStorageSettings::decryptValue($value);

        return match ($provider) {
            's3' => array_merge($config, [
                'key' => $decrypt($config['key'] ?? null),
                'secret' => $decrypt($config['secret'] ?? null),
            ]),
            'gcs' => array_merge($config, [
                'key_file' => $this->decodeKeyFile($decrypt($config['key_file'] ?? null)),
            ]),
            'gdrive' => array_merge($config, [
                'client_secret' => $decrypt($config['client_secret'] ?? null),
                'refresh_token' => $decrypt($config['refresh_token'] ?? null),
            ]),
            'azure' => array_merge($config, [
                'account_key' => $decrypt($config['account_key'] ?? null),
            ]),
            'bunny' => array_merge($config, [
                'access_key' => $decrypt($config['access_key'] ?? null),
            ]),
            'wasabi' => array_merge($config, [
                'key' => $decrypt($config['key'] ?? null),
                'secret' => $decrypt($config['secret'] ?? null),
            ]),
            default => $config,
        };
    }

    private function decodeKeyFile(?string $value): array
    {
        if (! $value) {
            return [];
        }

        $decoded = json_decode($value, true);

        return is_array($decoded) ? $decoded : [];
    }

    private function hasAll(array $config, array $keys): bool
    {
        foreach ($keys as $key) {
            if (empty($config[$key])) {
                return false;
            }
        }

        return true;
    }
}
