<?php

namespace App\Services;

use App\Jobs\DispatchMediaProcessing;
use App\Jobs\ProcessEventMedia;
use App\Models\Event;
use App\Models\EventMedia;
use Illuminate\Http\File;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Symfony\Component\Process\Process;
use ZipArchive;

class EventMediaService
{
    public function __construct(
        private CloudStorageManager $cloudStorage,
        private StorageUsageService $usageService,
        private PlanLimitService $planLimit
    ) {
    }

    /**
     * Store a batch of uploaded files with minimal per-file overhead.
     * Plan limits checked once. Bulk DB insert. Observers skipped for speed.
     */
    public function storeBatch(Event $event, array $files): int
    {
        $this->cloudStorage->registerDisks();
        $this->assertBatchPlanLimits($event, $files);

        $previewDisk = $this->previewDisk();
        $originalDisk = $this->originalDisk() ?: $previewDisk;
        $directory = $this->originalDirectory($event->id);
        $now = now();
        $uploaderId = auth()->id();
        $rows = [];
        $totalBytes = 0;

        foreach ($files as $file) {
            $originalName = $file->getClientOriginalName();
            $extension = $file->getClientOriginalExtension() ?: $file->extension();
            $fileName = $this->buildFileName($originalName, $extension);
            $mimeType = $file->getClientMimeType() ?: 'application/octet-stream';
            $size = (int) $file->getSize();

            $path = Storage::disk($originalDisk)->putFileAs($directory, $file, $fileName);

            $meta = ['original_disk' => $originalDisk];
            if ($uploaderId) {
                $meta['uploaded_by'] = $uploaderId;
            }

            $rows[] = [
                'event_id' => $event->id,
                'disk' => $previewDisk,
                'original_path' => $path,
                'file_name' => $originalName,
                'file_type' => $this->resolveFileType($mimeType, $originalName),
                'mime_type' => $mimeType,
                'size' => $size,
                'status' => 'pending',
                'meta' => json_encode($meta),
                'created_at' => $now,
                'updated_at' => $now,
            ];
            $totalBytes += $size;
        }

        if (empty($rows)) {
            return 0;
        }

        // Capture max ID before insert for reliable ID retrieval
        $beforeMaxId = (int) (EventMedia::max('id') ?? 0);

        // Bulk insert (1 query instead of N) — skip observers for speed
        EventMedia::insert($rows);

        // Update storage usage once for entire batch
        if ($totalBytes > 0) {
            $event->increment('storage_used_bytes', $totalBytes);
        }

        // Get inserted IDs by ID range (no timestamp race condition)
        $mediaIds = EventMedia::where('event_id', $event->id)
            ->where('id', '>', $beforeMaxId)
            ->where('status', 'pending')
            ->orderBy('id')
            ->pluck('id');

        // Single dispatch — bulk job dispatches individual jobs from queue worker
        DispatchMediaProcessing::dispatch($mediaIds->toArray());

        return count($rows);
    }

    /**
     * Extract a ZIP file and store all images/videos inside it.
     */
    public function storeZip(Event $event, UploadedFile $zipFile): int
    {
        $t0 = microtime(true);
        $this->cloudStorage->registerDisks();

        $zip = new ZipArchive();
        $tempDir = storage_path('app/tmp/zip-' . Str::random(12));

        if ($zip->open($zipFile->getRealPath()) !== true) {
            throw new \RuntimeException('Failed to open ZIP file.');
        }

        try {
            @mkdir($tempDir, 0755, true);
            $zip->extractTo($tempDir);
            $zip->close();
            $t1 = microtime(true);
            \Log::info('[ZIP] Extract done', ['seconds' => round($t1 - $t0, 2)]);

            // Collect valid image/video files from extracted contents
            $allowedExtensions = ['jpg', 'jpeg', 'png', 'webp', 'mp4'];
            $filePaths = [];
            $iterator = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator($tempDir, \FilesystemIterator::SKIP_DOTS)
            );

            foreach ($iterator as $fileInfo) {
                if (! $fileInfo->isFile()) {
                    continue;
                }
                $ext = strtolower($fileInfo->getExtension());
                if (in_array($ext, $allowedExtensions, true)) {
                    $filePaths[] = $fileInfo->getRealPath();
                }
            }

            if (empty($filePaths)) {
                return 0;
            }

            $t2 = microtime(true);
            \Log::info('[ZIP] Scan done', ['files' => count($filePaths), 'seconds' => round($t2 - $t1, 2)]);

            // Check plan limits once for entire ZIP
            $totalBytes = 0;
            $imageCount = 0;
            foreach ($filePaths as $fp) {
                $size = (int) filesize($fp);
                $totalBytes += $size;
                $ext = strtolower(pathinfo($fp, PATHINFO_EXTENSION));
                if ($ext !== 'mp4') {
                    $imageCount++;
                }
            }
            $this->planLimit->assertCanUpload($event, 'image', $totalBytes, $imageCount);
            $this->assertStorageLimit($totalBytes);

            // Store all files and bulk insert
            $previewDisk = $this->previewDisk();
            $originalDisk = $this->originalDisk() ?: $previewDisk;
            $directory = $this->originalDirectory($event->id);
            $now = now();
            $uploaderId = auth()->id();
            $rows = [];

            // Fast MIME type lookup by extension (avoid finfo_file per file)
            $mimeMap = [
                'jpg' => 'image/jpeg', 'jpeg' => 'image/jpeg',
                'png' => 'image/png', 'webp' => 'image/webp',
                'mp4' => 'video/mp4',
            ];

            // Check if original disk is local (enables rename optimization)
            $isLocalDisk = config('filesystems.disks.' . $originalDisk . '.driver') === 'local';
            $diskRoot = $isLocalDisk ? Storage::disk($originalDisk)->path('') : null;

            $t3 = microtime(true);
            \Log::info('[ZIP] Moving files start', ['disk' => $originalDisk, 'local' => $isLocalDisk]);

            foreach ($filePaths as $fp) {
                $originalName = basename($fp);
                $extension = strtolower(pathinfo($fp, PATHINFO_EXTENSION));
                $fileName = $this->buildFileName($originalName, $extension);
                $mimeType = $mimeMap[$extension] ?? 'application/octet-stream';
                $size = (int) filesize($fp);

                // Use rename() for local disk (O(1) inode move) instead of putFileAs() (full copy)
                if ($isLocalDisk) {
                    $targetDir = $diskRoot . str_replace('/', DIRECTORY_SEPARATOR, $directory);
                    if (!is_dir($targetDir)) {
                        @mkdir($targetDir, 0755, true);
                    }
                    rename($fp, $targetDir . DIRECTORY_SEPARATOR . $fileName);
                    $storedPath = $directory . '/' . $fileName;
                } else {
                    $file = new File($fp);
                    $storedPath = Storage::disk($originalDisk)->putFileAs($directory, $file, $fileName);
                }

                $meta = ['original_disk' => $originalDisk];
                if ($uploaderId) {
                    $meta['uploaded_by'] = $uploaderId;
                }

                $rows[] = [
                    'event_id' => $event->id,
                    'disk' => $previewDisk,
                    'original_path' => $storedPath,
                    'file_name' => $originalName,
                    'file_type' => $this->resolveFileType($mimeType, $originalName),
                    'mime_type' => $mimeType,
                    'size' => $size,
                    'status' => 'pending',
                    'meta' => json_encode($meta),
                    'created_at' => $now,
                    'updated_at' => $now,
                ];
            }

            $t4 = microtime(true);
            \Log::info('[ZIP] Move done', ['files' => count($rows), 'seconds' => round($t4 - $t3, 2)]);

            if (! empty($rows)) {
                // Capture max ID before insert for reliable ID retrieval
                $beforeMaxId = (int) (EventMedia::max('id') ?? 0);

                // Bulk insert
                foreach (array_chunk($rows, 500) as $chunk) {
                    EventMedia::insert($chunk);
                }

                $t5 = microtime(true);
                \Log::info('[ZIP] DB insert done', ['rows' => count($rows), 'seconds' => round($t5 - $t4, 2)]);

                // Update storage once
                if ($totalBytes > 0) {
                    $event->increment('storage_used_bytes', $totalBytes);
                }

                $t5a = microtime(true);
                \Log::info('[ZIP] Storage increment done', ['seconds' => round($t5a - $t5, 2)]);

                // Get inserted IDs by ID range (no timestamp race condition)
                $mediaIds = EventMedia::where('event_id', $event->id)
                    ->where('id', '>', $beforeMaxId)
                    ->where('status', 'pending')
                    ->orderBy('id')
                    ->pluck('id');

                $t5b = microtime(true);
                \Log::info('[ZIP] ID query done', ['ids' => $mediaIds->count(), 'seconds' => round($t5b - $t5a, 2)]);

                // Single dispatch — bulk job dispatches individual jobs from queue worker
                \Log::info('[ZIP] About to dispatch DispatchMediaProcessing', ['id_count' => $mediaIds->count(), 'queue_driver' => config('queue.default')]);
                DispatchMediaProcessing::dispatch($mediaIds->toArray());

                $t6 = microtime(true);
                \Log::info('[ZIP] Dispatch done', ['jobs' => $mediaIds->count(), 'seconds' => round($t6 - $t5b, 2)]);
            }

            $totalTime = round(microtime(true) - $t0, 2);
            \Log::info('[ZIP] COMPLETE', ['files' => count($rows), 'total_seconds' => $totalTime]);

            return count($rows);
        } finally {
            // Always cleanup temp directory
            $cleanStart = microtime(true);
            $this->deleteDirectory($tempDir);
            \Log::info('[ZIP] Cleanup done', ['seconds' => round(microtime(true) - $cleanStart, 2)]);
        }
    }

    private function deleteDirectory(string $dir): void
    {
        if (! is_dir($dir)) {
            return;
        }
        $items = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($dir, \FilesystemIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::CHILD_FIRST
        );
        foreach ($items as $item) {
            $item->isDir() ? @rmdir($item->getRealPath()) : @unlink($item->getRealPath());
        }
        @rmdir($dir);
    }

    /**
     * Store a single uploaded file (used by chunk upload path).
     */
    public function storeUploadedFile(Event $event, UploadedFile $file): EventMedia
    {
        $this->cloudStorage->registerDisks();
        $originalName = $file->getClientOriginalName();
        $extension = $file->getClientOriginalExtension() ?: $file->extension();
        $fileType = $this->resolveFileType($file->getClientMimeType(), $originalName);
        $this->assertPlanLimits($event, $fileType, (int) $file->getSize());

        $previewDisk = $this->previewDisk();
        $originalDisk = $this->originalDisk() ?: $previewDisk;
        $fileName = $this->buildFileName($originalName, $extension);
        $directory = $this->originalDirectory($event->id);

        $path = Storage::disk($originalDisk)->putFileAs($directory, $file, $fileName);

        $media = $this->createMediaRecord($event, [
            'disk' => $previewDisk,
            'original_path' => $path,
            'file_name' => $originalName,
            'mime_type' => $file->getClientMimeType(),
            'size' => (int) $file->getSize(),
            'meta' => ['original_disk' => $originalDisk],
        ]);

        ProcessEventMedia::dispatch($media->id);

        return $media;
    }

    public function storeLocalFile(Event $event, string $path, ?string $originalName = null): ?EventMedia
    {
        if (! is_file($path)) {
            return null;
        }

        $this->cloudStorage->registerDisks();

        $previewDisk = $this->previewDisk();
        $originalDisk = $this->originalDisk() ?: $previewDisk;
        $file = new File($path);
        $extension = $file->getExtension();
        $fileName = $this->buildFileName($originalName ?: $file->getFilename(), $extension);
        $directory = $this->originalDirectory($event->id);
        $fileType = $this->resolveFileType($file->getMimeType() ?: 'application/octet-stream', $originalName ?: $file->getFilename());

        $this->assertPlanLimits($event, $fileType, (int) $file->getSize());

        $storedPath = Storage::disk($originalDisk)->putFileAs($directory, $file, $fileName);

        $media = $this->createMediaRecord($event, [
            'disk' => $previewDisk,
            'original_path' => $storedPath,
            'file_name' => $originalName ?: $file->getFilename(),
            'mime_type' => $file->getMimeType(),
            'size' => (int) $file->getSize(),
            'meta' => ['original_disk' => $originalDisk],
        ]);

        ProcessEventMedia::dispatch($media->id);

        return $media;
    }

    private function createMediaRecord(Event $event, array $payload): EventMedia
    {
        $dimensions = $payload['dimensions'] ?? null;
        $mimeType = $payload['mime_type'] ?: 'application/octet-stream';
        $meta = is_array($payload['meta'] ?? null) ? $payload['meta'] : [];
        $uploaderId = auth()->id();
        if ($uploaderId) {
            $meta['uploaded_by'] = $uploaderId;
        }

        return EventMedia::create([
            'event_id' => $event->id,
            'disk' => $payload['disk'],
            'original_path' => $payload['original_path'],
            'optimized_path' => $payload['optimized_path'] ?? null,
            'thumbnail_path' => $payload['thumbnail_path'] ?? null,
            'file_name' => $payload['file_name'],
            'file_type' => $this->resolveFileType($mimeType, $payload['file_name']),
            'mime_type' => $mimeType,
            'size' => $payload['size'],
            'file_hash' => $payload['file_hash'] ?? null, // FIX #18
            'width' => $dimensions['width'] ?? null,
            'height' => $dimensions['height'] ?? null,
            'status' => 'pending',
            'meta' => $meta ?: null,
        ]);
    }

    /**
     * FIX #18: Check for duplicate files in the event
     */
    private function checkDuplicate(Event $event, string $hash): ?EventMedia
    {
        if (!config('events.media.detect_duplicates', true)) {
            return null;
        }

        return $event->media()
            ->where('file_hash', $hash)
            ->first();
    }

    private function buildFileName(string $originalName, ?string $extension): string
    {
        // UUID only — no extension, no original name trace.
        // Prevents casual file browsing; mime_type stored in DB for serving.
        return (string) Str::uuid();
    }

    private function resolveFileType(string $mimeType, string $fileName): string
    {
        if (str_starts_with($mimeType, 'video/')) {
            return 'video';
        }

        $extension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
        if (in_array($extension, ['mp4'], true)) {
            return 'video';
        }

        return 'image';
    }

    private function readDimensions(string $path): ?array
    {
        if (! is_file($path)) {
            return null;
        }

        $info = @getimagesize($path);
        if (! $info) {
            return null;
        }

        return [
            'width' => $info[0] ?? null,
            'height' => $info[1] ?? null,
        ];
    }

    private function originalDirectory(int $eventId): string
    {
        return $this->cloudStorage->applyPrefix('events/' . $eventId . '/original');
    }

    private function previewDisk(): string
    {
        return $this->cloudStorage->previewDiskName();
    }

    private function originalDisk(): string
    {
        return $this->cloudStorage->originalDiskName();
    }

    private function copyToPreviewDisk(string $originalDisk, string $originalPath, string $previewDisk, int $eventId): ?string
    {
        if ($originalDisk === $previewDisk) {
            return null;
        }

        $previewPath = $this->buildVariantPath($eventId, $originalPath, 'optimized');
        $previewStorage = Storage::disk($previewDisk);
        $previewStorage->makeDirectory(dirname($previewPath));

        $stream = Storage::disk($originalDisk)->readStream($originalPath);
        if (! $stream) {
            return null;
        }

        $previewStorage->put($previewPath, $stream);
        if (is_resource($stream)) {
            fclose($stream);
        }

        return $previewPath;
    }

    private function buildVariantPath(int $eventId, string $originalPath, string $variant): string
    {
        return $this->cloudStorage->buildVariantPath($eventId, $originalPath, $variant);
    }

    private function assertStorageLimit(int $bytes): void
    {
        $this->usageService->assertCanStore($bytes);
    }

    private function assertPlanLimits(Event $event, string $fileType, int $bytes): void
    {
        $this->planLimit->assertCanUpload($event, $fileType, $bytes);
        $this->assertStorageLimit($bytes);
    }

    /**
     * Check plan limits once for an entire batch of files.
     */
    private function assertBatchPlanLimits(Event $event, array $files): void
    {
        $imageCount = 0;
        $totalBytes = 0;

        foreach ($files as $file) {
            $fileType = $this->resolveFileType($file->getClientMimeType(), $file->getClientOriginalName());
            if ($fileType === 'image') {
                $imageCount++;
            }
            $totalBytes += (int) $file->getSize();
        }

        // Check with total counts in a single call
        $this->planLimit->assertCanUpload($event, 'image', $totalBytes, $imageCount);
        $this->assertStorageLimit($totalBytes);
    }

    /**
     * Validate video duration to enforce upload limits
     */
    private function validateVideoDuration(string $path): void
    {
        $maxDuration = (int) config('events.media.max_video_duration', 600); // 10 min default
        if ($maxDuration <= 0) {
            return;
        }

        $ffprobe = config('ffmpeg.ffprobe_binary', 'ffprobe');

        try {
            $process = new Process([
                $ffprobe, '-v', 'error',
                '-show_entries', 'format=duration',
                '-of', 'default=noprint_wrappers=1:nokey=1',
                $path,
            ]);
            $process->setTimeout(15);
            $process->run();

            if ($process->isSuccessful()) {
                $duration = (float) trim($process->getOutput());
                if ($duration > $maxDuration) {
                    throw new \RuntimeException(
                        "Video duration ({$duration}s) exceeds maximum ({$maxDuration}s). Please upload a shorter video."
                    );
                }
            }
        } catch (\RuntimeException $e) {
            throw $e;
        } catch (\Throwable $e) {
            // ffprobe not available - skip validation
        }
    }

    /**
     * FIX #3: Validate image dimensions to prevent memory exhaustion
     */
    private function validateImageDimensions($file): void
    {
        $path = $file instanceof \Illuminate\Http\UploadedFile ? $file->getRealPath() : $file;

        $dimensions = @getimagesize($path);
        if ($dimensions === false) {
            return; // Not an image or unreadable - will be caught later
        }

        [$width, $height] = $dimensions;

        // Max 10,000 pixels per side (configurable)
        $maxDimension = (int) config('events.media.max_dimension', 10000);

        // Max 100 megapixels total (configurable)
        $maxPixels = (int) config('events.media.max_pixels', 100000000);

        if ($width > $maxDimension || $height > $maxDimension) {
            throw new \RuntimeException(
                "Image dimensions ({$width}x{$height}px) exceed maximum ({$maxDimension}px per side). " .
                "Please resize the image before uploading."
            );
        }

        $totalPixels = $width * $height;
        if ($totalPixels > $maxPixels) {
            $maxMP = round($maxPixels / 1000000, 1);
            $actualMP = round($totalPixels / 1000000, 1);
            throw new \RuntimeException(
                "Image size ({$actualMP} megapixels) exceeds maximum ({$maxMP} megapixels). " .
                "Please use a smaller image."
            );
        }
    }
}
