<?php

namespace App\Services;

use App\Models\Event;
use App\Models\EventMedia;

/**
 * FIX #14: Per-Event Storage Quota Service
 */
class EventStorageService
{
    /**
     * Get current storage usage for an event
     */
    public function getUsage(Event $event): int
    {
        return $event->media()->sum('size') ?? 0;
    }

    /**
     * Update the cached storage usage for an event
     */
    public function updateUsage(Event $event): void
    {
        $usage = $this->getUsage($event);
        $event->update(['storage_used_bytes' => $usage]);
    }

    /**
     * Check if event can upload a file of given size
     */
    public function canUpload(Event $event, int $fileSize): bool
    {
        // No quota set = unlimited
        if (!$event->storage_quota_bytes) {
            return true;
        }

        $currentUsage = $event->storage_used_bytes ?? $this->getUsage($event);
        return ($currentUsage + $fileSize) <= $event->storage_quota_bytes;
    }

    /**
     * Assert that event can upload a file of given size
     *
     * @throws \RuntimeException
     */
    public function assertCanUpload(Event $event, int $fileSize): void
    {
        if (!$this->canUpload($event, $fileSize)) {
            $quotaMB = round($event->storage_quota_bytes / 1024 / 1024, 1);
            $usedMB = round(($event->storage_used_bytes ?? 0) / 1024 / 1024, 1);
            $fileMB = round($fileSize / 1024 / 1024, 1);

            throw new \RuntimeException(
                "Event storage quota exceeded. Used {$usedMB}MB of {$quotaMB}MB quota. Cannot upload {$fileMB}MB file."
            );
        }
    }

    /**
     * Get storage usage statistics for an event
     */
    public function getStats(Event $event): array
    {
        $used = $event->storage_used_bytes ?? $this->getUsage($event);
        $quota = $event->storage_quota_bytes;

        return [
            'used_bytes' => $used,
            'quota_bytes' => $quota,
            'used_mb' => round($used / 1024 / 1024, 2),
            'quota_mb' => $quota ? round($quota / 1024 / 1024, 2) : null,
            'percentage' => $quota ? round(($used / $quota) * 100, 1) : null,
            'available_bytes' => $quota ? max(0, $quota - $used) : null,
            'available_mb' => $quota ? round(max(0, $quota - $used) / 1024 / 1024, 2) : null,
        ];
    }
}
