<?php

namespace App\Services;

use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

class ExportService
{
    public function download(string $format, string $filename, Builder $query, array $columns, callable $map, string $title, string $description)
    {
        $format = strtolower($format);

        return match ($format) {
            'csv' => $this->exportCsv($query, $columns, $map, $filename . '.csv'),
            'xlsx' => $this->exportXlsx($query->get()->map($map), $columns, $filename . '.xlsx'),
            'pdf' => $this->exportPdf($query->get()->map($map), $columns, $title, $description, $filename . '.pdf'),
            default => abort(404),
        };
    }

    private function exportCsv(Builder $query, array $columns, callable $map, string $filename)
    {
        return response()->streamDownload(function () use ($query, $columns, $map) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, array_map(fn ($column) => $column['label'], $columns));

            $query->chunkById(200, function ($items) use ($handle, $columns, $map) {
                foreach ($items as $item) {
                    $row = $map($item);
                    fputcsv($handle, $this->formatRow($columns, $row));
                }
            });

            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    private function exportXlsx(Collection $rows, array $columns, string $filename)
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $columnIndex = 1;
        foreach ($columns as $column) {
            $sheet->setCellValue([$columnIndex, 1], $column['label']);
            $columnIndex++;
        }

        $rowIndex = 2;
        foreach ($rows as $row) {
            $columnIndex = 1;
            foreach ($columns as $column) {
                $sheet->setCellValue([$columnIndex, $rowIndex], $row[$column['key']] ?? '');
                $columnIndex++;
            }
            $rowIndex++;
        }

        $writer = new Xlsx($spreadsheet);

        return response()->streamDownload(function () use ($writer) {
            $writer->save('php://output');
        }, $filename, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    private function exportPdf(Collection $rows, array $columns, string $title, string $description, string $filename)
    {
        $pdf = Pdf::loadView('admin.reports.pdf', [
            'report' => [
                'title' => $title,
                'description' => $description,
            ],
            'columns' => $columns,
            'rows' => $rows,
            'generatedAt' => now(),
        ])->setPaper('a4', 'landscape');

        return $pdf->download($filename);
    }

    private function formatRow(array $columns, array $row): array
    {
        return array_map(function ($column) use ($row) {
            return $row[$column['key']] ?? '';
        }, $columns);
    }
}
